 /*
 * $Id: pyxasw.c,v 1.3 2003/02/14 16:11:11 lsmithso Exp $
 * pyxasw.c: A Python Embed/Extension program that implements an XA
 * Switch interface, as defined by the The Open Group XA
 * specification.
 *
 *
 * Author: L. Smithson (lsmithson@open-networks.co.uk)
 *
 * DISCLAIMER
 * You are free to use this code in any way you like, subject to the
 * standard Python disclaimers & copyrights. I make no representations
 * about the suitability of this software for any purpose. It is
 * provided "AS-IS" without warranty of any kind, either express or
 * implied. So there.
 *
 *
 * Pyxasw is initialized with a XA switch instance from a real
 * Resource Manager (such as Oracle), and wraps each of the upcalls in
 * Python extensions (ext*). A new XA switch instance is returned
 * whose upcalls invoke functions (emb*) in an embedded Python
 * script. These functions may call the real RM XA functions via the
 * extensions, or they may do their own thing, or any combination of
 * the two.  If the embedded script does not exist, or a function is
 * not defined, then the real (wrapped) XA function is called.
 *
 *
 * Pyxasw may be invoked as the result of a Python script (e.g., pymqi
 * begin or commit calls), or a non-Python program (e.g., a Resource
 * Manager, or a native MQ 'C' application).
 *
 * If called from a Python script, the Interpreter is already
 * initialized, but it is assumed that the global interpreter lock is
 * not held, and that there is no thread context (this is the case for
 * pymqi, and is a reasonable assumption for any other Python
 * extension that make blocking transactional calls). In this case, a
 * new sub-interpreter is started (using Py_NewInterpreter) and its
 * new thread state is saved in a static variable. Before any embedded
 * Python API calls are made, the global interpreter lock is acquired,
 * and the thread state is set to that saved from the call to
 * Py_NewInterpreter. When the embedded Python call returns, the
 * current thread state is zapped and the global lock is released.
 *
 * If called from a non-Python program, the interpreter is initialized
 * as normal. When embedded Python API calls are made, the gobal
 * interpreter lock is already held, and there is no need to set up a
 * thread state.
 *
 * The extension functions are assumed to follow normal conventions,
 * that is they are called with the global interpreter lock held and
 * with a thread state. The wrapped XA calls are bracketed by
 * Py_BEGIN_ALLOW_THREADS/Py_END_ALLOW_THREADS. This allows other
 * Python threads to run if an XA upcall blocks.
 *
 *
 */

static char __version__[] = "0.1";
static char  mod_doc[] = "TBD!";
#include <Python.h>

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>

#include "xa.h"

static struct xa_switch_t myXaSw;        /* XA Switch wrapper returned to the RM*/
static struct xa_switch_t *wrappedXaSw;  /* Wrapped XA Switch - the real thing. */
static PyObject *scriptDict = 0;         /* Embedded script dictionary */
static PyThreadState *privThreadState = 0;   /* Maybe our private thread state */

#define ERR_BADEMBED 100                 /* The error we return after a bad embed call */


/*
 * These calls bracket embedded Python calls. They ensure the
 * interpreter lock is held, and that a thread state exists.
 */
static void PrePyEmbedCall(void) {
  if (privThreadState) {
    /* Acuires the global lock */
    PyEval_RestoreThread(privThreadState);
  }
}

static void PostPyEmbedCall(void) {
  if (privThreadState) {
    /* Release the global lock */
    PyEval_SaveThread();
  }
}

#if 0
/* Used for debugging during RM calls when there is no tty. */
static void myxalog(const char *fmt, ...)
{
  FILE *fp;
  va_list ap;
  int pid;
  pthread_t threadId;

  va_start(ap, fmt);
  fp = fopen("/tmp/xalog.txt", "a+");
  pid = getpid();
  threadId = pthread_self();
  fprintf(fp, "%d:%ld ", pid, threadId);
  vfprintf(fp, fmt, ap);
  fclose(fp);
}
#endif

/*
 * XID <-> Python string manipulation. XID is passed between
 * Python & this module as opaque s# strings. This is just me being
 * paranoid about copying user supplied buffers.
 */
static void makeXidFromString(char *xids, int xidLength, XID *xidRef) {
  int len = (xidLength < sizeof(XID)) ? xidLength : sizeof(XID);
  memcpy(xidRef, xids, len);
}


/*
 * Python Extension methods that make real RM XA Switch upcalls. These
 * may be called by the embedded Python script, or after an explicit
 * import.
 */

/*
 * Factored out xa call of the form xacall(XID*, int, long). 
 */
typedef int (*XACall_XidIntLong_t)(XID*, int, long);

static PyObject *extXa_CallXidIntLong(XACall_XidIntLong_t fn, PyObject *self, PyObject *args) {
  char *xids;
  int xidLength;
  int rmid, retVal;
  long flags;
  XID xid;
  
  if(!PyArg_ParseTuple(args, "s#il", &xids, &xidLength, &rmid, &flags )) {
    return NULL;
  }
  makeXidFromString(xids, xidLength, &xid);
  Py_BEGIN_ALLOW_THREADS
  retVal = fn(&xid, rmid, flags);
  Py_END_ALLOW_THREADS

  return Py_BuildValue("i", retVal);
}



static PyObject* extXaOpen(PyObject *self, PyObject *args) {
  char *openString;
  long flags;
  int rmid, retVal;
  
  if(!PyArg_ParseTuple(args, "sil", &openString, &rmid, &flags )) {
    return NULL;
  }
  Py_BEGIN_ALLOW_THREADS
  retVal = wrappedXaSw->xa_open_entry(openString, rmid, flags);
  Py_END_ALLOW_THREADS
  return Py_BuildValue("i", retVal);
}


static PyObject* extXaClose(PyObject *self, PyObject *args) {
  char *closeString;
  long flags;
  int rmid, retVal;
  
  if(!PyArg_ParseTuple(args, "sil", &closeString, &rmid, &flags )) {
    return NULL;
  }
  Py_BEGIN_ALLOW_THREADS
  retVal = wrappedXaSw->xa_close_entry(closeString, rmid, flags);
  Py_END_ALLOW_THREADS
  return Py_BuildValue("i", retVal);
}


static PyObject* extXaStart(PyObject *self, PyObject *args) {
  return extXa_CallXidIntLong(wrappedXaSw->xa_start_entry, self, args);
}


static PyObject* extXaEnd(PyObject *self, PyObject *args) {
  return extXa_CallXidIntLong(wrappedXaSw->xa_end_entry, self, args);
}


static PyObject* extXaRollback(PyObject *self, PyObject *args) {
  return extXa_CallXidIntLong(wrappedXaSw->xa_rollback_entry, self, args);
}


static PyObject* extXaPrepare(PyObject *self, PyObject *args) {
  return extXa_CallXidIntLong(wrappedXaSw->xa_prepare_entry, self, args);
}


static PyObject* extXaCommit(PyObject *self, PyObject *args) {
  return extXa_CallXidIntLong(wrappedXaSw->xa_commit_entry, self, args);
}


static PyObject* extXaRecover(PyObject *self, PyObject *args) {
  char *xids;
  int xidLength;
  long count, flags;
  int rmid, retVal;
  
  if(!PyArg_ParseTuple(args, "s#lil", &xids, &xidLength, &count, &rmid, &flags )) {
    return NULL;
  }
  /*
   * xids is really an array xid[count], just pass the whole buffer
   * along, without safe copying to an xid.
   */ 
  Py_BEGIN_ALLOW_THREADS
  retVal = wrappedXaSw->xa_recover_entry((XID *)xids, count, rmid, flags);
  Py_END_ALLOW_THREADS
  return Py_BuildValue("i", retVal);
}


static PyObject* extXaForget(PyObject *self, PyObject *args) {
  return extXa_CallXidIntLong(wrappedXaSw->xa_commit_entry, self, args);
}


static PyObject* extXaComplete(PyObject *self, PyObject *args) {
  int handle, xarv;
  long flags;
  int rmid, retVal;
  
  if(!PyArg_ParseTuple(args, "iil", &handle, &rmid, &flags )) {
    return NULL;
  }
  Py_BEGIN_ALLOW_THREADS
  retVal = wrappedXaSw->xa_complete_entry(&handle, &xarv, rmid, flags);
  Py_END_ALLOW_THREADS
  return Py_BuildValue("ii", retVal, xarv);
}


/*
 * Additional methods to encode/decode XID as formatted strings. Not
 * part of the XA spec but useful for scripts that want to print XIDs.
 */

/*
 * Format each byte of in (length n) as a hex char in out, Returns
 * length of formatted string
 */

static int xidFmtPart(char *in, int len, char *out) {
  int i;
  int o = 0;
  out[o] = 0;
  for (i = 0; i < len; i++) {
    sprintf(out + o, "%02x", (unsigned char)in[i] & 0xff);
    o += 2;
  }
  return o;
}

static PyObject* extXidToString(PyObject *self, PyObject *args) {
  char *xids;
  int xidLength;
  XID xid;
  char *strXid = "";
  char res[130];
  int offs;
    
  if(!PyArg_ParseTuple(args, "s#", &xids, &xidLength)) {
    return NULL;
  }
  makeXidFromString(xids, xidLength, &xid);
  if (xid.formatID != -1) {
    offs = xidFmtPart(xid.data, xid.gtrid_length, res);
    res[offs] = ':';
    xidFmtPart(xid.data + xid.gtrid_length, xid.bqual_length, res + offs + 1);
    strXid = res;
  }
  return Py_BuildValue("s", strXid);
}


static PyObject* extStringToXid(PyObject *self, PyObject *args) {
  XID xid;
  char *xids;
  int i;
  char tmpb[] = {0, 0, 0};
  long t;
  
  if(!PyArg_ParseTuple(args, "s", &xids)) {
    return NULL;
  }
  for (i = 0; i < strlen(xids); i += 2) {
    if (xids[i] == ':') {
      xid.gtrid_length = i/2;
      i--;
    } else {
      tmpb[0] = xids[i]; tmpb[1] = xids[i + 1];
      t = strtol(tmpb, NULL, 16);
      xid.data[i/2] = (char)t;
    }
    xid.bqual_length = (i/2) - xid.bqual_length;
  }
  return Py_BuildValue("s#", (char *)&xid, sizeof(xid));
}


static PyMethodDef extXaMethods[] = {
  {"rmopen", extXaOpen, METH_VARARGS, "Real RM XA Open call."},
  {"rmclose", extXaClose, METH_VARARGS, "Real RM XA Close call."},
  {"rmstart", extXaStart, METH_VARARGS, "Real RM XA Start call."},
  {"rmend", extXaEnd, METH_VARARGS, "Real RM XA End call."},
  {"rmrollback", extXaRollback, METH_VARARGS, "Real RM XA Rollback call."},
  {"rmprepare", extXaPrepare, METH_VARARGS, "Real RM XA Prepare call."},
  {"rmcommit", extXaCommit, METH_VARARGS, "Real RM XA Commit call."},
  {"rmrecover", extXaRecover, METH_VARARGS, "Real RM XA Recover call."},
  {"rmforget", extXaForget, METH_VARARGS, "Real RM XA Forget call."},
  {"rmcomplete", extXaComplete, METH_VARARGS, "Real RM XA Complete call."},
  {"rmrecover", extXaComplete, METH_VARARGS, "Real RM XA Complete call."},
  {"xid2str", extXidToString, METH_VARARGS, "XID to string."},
  {"str2xid", extStringToXid, METH_VARARGS, "String to XID."},
  
  {NULL, NULL, 0, NULL}
};


/*
 * Initialisation for a module import outside of the XA switch
 * load. This allows a script to be loaded & at least syntax checked,
 * but probably little else.
 */

void initpyxasw(void) {
  PyObject *mod, *dict;
  mod = Py_InitModule3("pyxasw", extXaMethods, mod_doc);
  dict = PyModule_GetDict(mod);
  PyDict_SetItemString(dict,"__version__", PyString_FromString(__version__));
  PyDict_SetItemString(dict, "__doc__", PyString_FromString(mod_doc)); 
}



/*
 * Embedded calls. These are xa calls made by a TM through the xa
 * switch returned by pyxasw_Wrapper. They result in calls to an
 * embedded script function, if one can be found. If the function is
 * not defined or not callable, or the script could not be loaded,
 * then the wrapped xa switch func. is called.
 */


/*
 * Handle embedded script runtime errors
 */

void embScriptError(char *what) {
  PyObject *etype, *evalue, *etrace;
  PyErr_Fetch(&etype, &evalue, &etrace);
  /* Much more to do here */
}

/*
 * Factored out common fn.
 */
int embXa_CallXidIntLong(char *pyFname, XACall_XidIntLong_t fnp, XID *xid, int rmid, long flags) {
  PyObject *funcObj, *rsltObj;
  int retVal;
  int dfltCallReq = 1;
  if (scriptDict) {
    PrePyEmbedCall();
    funcObj = PyDict_GetItemString(scriptDict, pyFname);
    if (funcObj && PyCallable_Check(funcObj)) {
      dfltCallReq = 0;
      rsltObj = PyObject_CallFunction(funcObj, "(s#il)", xid, sizeof(XID), rmid, flags);
      if (rsltObj) {
	retVal = PyInt_AsLong(rsltObj);
      } else {
	embScriptError(pyFname);
	retVal = ERR_BADEMBED;
      }
    }
    PostPyEmbedCall();
  }
  if (dfltCallReq) { 
    retVal = fnp(xid, rmid, flags);
  }
  return retVal;
}


		    
int embXaOpen(char *openString, int rmid, long flags) {
  int retVal = 0;
  PyObject *funcObj, *rsltObj;
  int dfltCallReq = 1;
  
  if (scriptDict) {
    PrePyEmbedCall();
    funcObj = PyDict_GetItemString(scriptDict, "xa_open");
    if (funcObj && PyCallable_Check(funcObj)) {
      dfltCallReq = 0;
      rsltObj = PyObject_CallFunction(funcObj, "(sil)", openString, rmid, flags);
      if (rsltObj) {
	retVal = PyInt_AsLong(rsltObj);
      } else {
	embScriptError("open");
	retVal = ERR_BADEMBED;
      }
    }
    PostPyEmbedCall();
  }
  if (dfltCallReq) {
    retVal = wrappedXaSw->xa_open_entry(openString, rmid, flags);
  }
  return retVal;
}


int embXaClose(char *closeString, int rmid, long flags) {
  int retVal = 0;
  PyObject *funcObj, *rsltObj;
  int dfltCallReq = 1;

  if (scriptDict) {
    PrePyEmbedCall();
    funcObj = PyDict_GetItemString(scriptDict, "xa_close");
    if (funcObj && PyCallable_Check(funcObj)) {
      dfltCallReq = 0;
      rsltObj = PyObject_CallFunction(funcObj, "(sil)", closeString, rmid, flags);
      if (rsltObj) {
	retVal = PyInt_AsLong(rsltObj);
      } else {
	embScriptError("close");
	retVal = ERR_BADEMBED;
      }
    }
    PostPyEmbedCall();
  }
  if (dfltCallReq) {
    retVal = wrappedXaSw->xa_close_entry(closeString, rmid, flags);
  }
  return retVal;
}


int embXaStart(XID *xid, int rmid, long flags) {
  return embXa_CallXidIntLong("xa_start", wrappedXaSw->xa_start_entry, xid, rmid, flags);
}

int embXaEnd(XID *xid, int rmid, long flags) {
  return embXa_CallXidIntLong("xa_end", wrappedXaSw->xa_end_entry, xid, rmid, flags);
}

int embXaRollback(XID *xid, int rmid, long flags) {
  return embXa_CallXidIntLong("xa_rollback", wrappedXaSw->xa_rollback_entry, xid, rmid, flags);
}

int embXaPrepare(XID *xid, int rmid, long flags) {
  return embXa_CallXidIntLong("xa_prepare", wrappedXaSw->xa_prepare_entry, xid, rmid, flags);
}

int embXaCommit(XID *xid, int rmid, long flags) {
  return embXa_CallXidIntLong("xa_commit", wrappedXaSw->xa_commit_entry, xid, rmid, flags);
}


int embXaRecover(XID *xid, long count, int rmid, long flags) {
  int retVal = 0;
  PyObject *funcObj, *rsltObj;
  int dfltCallReq = 1;

  if (scriptDict) {
    PrePyEmbedCall();
    funcObj = PyDict_GetItemString(scriptDict, "xa_recover");
    if (funcObj && PyCallable_Check(funcObj)) {
      /*
       * xid here is actually an array of XID[count], so make sure all
       * of the array is passed to script.
       */
      dfltCallReq = 0;
      rsltObj = PyObject_CallFunction(funcObj, "(s#lil)", xid, sizeof(xid) * count, rmid, flags);
      if (rsltObj) {
	retVal = PyInt_AsLong(rsltObj);
      } else {
	embScriptError("recover");
	retVal = ERR_BADEMBED;
      }
      PostPyEmbedCall();
    }
  }
  if (dfltCallReq) {
    retVal = wrappedXaSw->xa_recover_entry(xid, count, rmid, flags);
  }
  return retVal;
}
 
int embXaForget(XID *xid, int rmid, long flags) {
  return embXa_CallXidIntLong("xa_forget", wrappedXaSw->xa_forget_entry, xid, rmid, flags);
}

int embXaComplete(int *handle, int *xarv, int rmid, long flags) {
  int retVal = 0;
  PyObject *funcObj, *rsltObj;
  int dfltCallReq = 1;

  if (scriptDict) {
    PrePyEmbedCall();
    funcObj = PyDict_GetItemString(scriptDict, "xa_complete");
    if (funcObj && PyCallable_Check(funcObj)) {
      dfltCallReq = 0;
      rsltObj = PyObject_CallFunction(funcObj, "(iil)", *handle, rmid, flags);  
      if (rsltObj) {
	/* FIXME: Should return a tuple of (retval, xarv) */
	retVal = PyInt_AsLong(rsltObj);
      } else {
	embScriptError("complete");
	retVal = ERR_BADEMBED;
      }
      PostPyEmbedCall();
    }
  }
  if (dfltCallReq) {
    retVal = wrappedXaSw->xa_complete_entry(handle, xarv, rmid, flags);
  }
  return retVal;
}


/*
 * This function is called by the TM XA program to pass the real RM XA
 * switch for Python wrapping. It returns a new XA switch that the TM
 * can use to drive the embedded Python XA calls.
 */

struct xa_switch_t *pyxasw_Wrapper(struct xa_switch_t *wrapee, char *modName) {
  PyObject *modNameObj = 0, *moduleObject = 0;
  char *tmpArgs[1];
  static int alreadyCalled = 0;
  
  /*
   * If already called in the current process, just return
   */
  if (alreadyCalled) {
    return &myXaSw;
  }
  
  /*
   * Setup default XA Switch. This is just a straight copy of the
   * wrapee. 
   */
  memcpy(&myXaSw, wrapee, sizeof(myXaSw));
  wrappedXaSw = wrapee;
  
  /*
   * This code can be called by non-Python scripts (by a TM or
   * non-Python app.) or Python scripts using pymqi, dco2 etc. So at
   * this point the Python interpreter may or may not be
   * initialised.
   */
  tmpArgs[0] = modName;
  if (!Py_IsInitialized()) {
    /*
     * No interpreter running, so start one, setting up & acquiring
     * the global lock & thread state etc.
     */
    Py_Initialize();
    PySys_SetArgv(1, tmpArgs);
  } else {
    /*
     * If the interp. is already running, start a sub-Interpreter to
     * handle all API/Script calls. The new thread state is saved. 
     */
    privThreadState = Py_NewInterpreter();
    PySys_SetArgv(1, tmpArgs);
  }

  /*
   * Initialise the extension, re-use the standard Python module
   * init. function above.
   */
  initpyxasw();

  /*
   * Try and import the embedded module. If it can be loaded, override
   * the XA switch with the embedded script function callers.
   */
  modNameObj = PyString_FromString(modName);
  if (modNameObj) {
    moduleObject = PyImport_Import(modNameObj);
    if (moduleObject) {
      scriptDict = PyModule_GetDict(moduleObject);
      /* Setup new xa callbacks */
      myXaSw.xa_open_entry = embXaOpen;
      myXaSw.xa_close_entry = embXaClose;
      myXaSw.xa_start_entry = embXaStart;
      myXaSw.xa_end_entry = embXaEnd;
      myXaSw.xa_prepare_entry = embXaPrepare;
      myXaSw.xa_commit_entry = embXaCommit;
      myXaSw.xa_rollback_entry = embXaRollback;
      myXaSw.xa_recover_entry = embXaRecover;
      myXaSw.xa_forget_entry = embXaForget;
      myXaSw.xa_complete_entry = embXaComplete;

    }
  }

  alreadyCalled = 1;
  return &myXaSw;
}
