# -*- coding: iso-8859-1 -*-
__version__ = '1.0'

from common import level_rule
from vms import starlet
from vms.rtl import lib
from vms import ssdef, kgbdef, ossdef, dcdef, dvidef
from vms import itemList
from secrules import get_security
from secrules import user_exists
from DeviceScan import DeviceScan

ids = (('BATCH',0x80000001), 
       ('DIALUP',0x80000002), 
       ('INTERACTIVE',0x80000003), 
       ('LOCAL',0x80000004), 
       ('NETWORK',0x80000005),
       ('REMOTE',0x80000006))

@level_rule(2)
def rule0701(fo, fmt):
    """ The OpenVMS system predefines fixed values for all system identifiers.  
If the system identifier values are changed it implies unnecessary tampering 
with system resources."""

    if not fmt:
        print>>fo
        print>>fo, "RULE 0701"
        print>>fo, "---------"

    for n, v in ids:
        try:
            s, idn, idv, ida = starlet.idtoasc(v)
            if idn != n:
                if fmt:
                    print>>fo, '07012', '%s %x' % (n, v)
                else:
                    print>>fo, '%s %x' % (n, v)
        except VMSError, e:
            if e.errno != ssdef.SS__NOSUCHID:
                print>>fo, e
            if fmt:
                print>>fo, '07012', '%s %x' % (n, v)
            else:
                print>>fo, '%s %x' % (n, v)

@level_rule(2)
def rule0702(fo, fmt):
    """ If a user is granted one of these identifiers, it may provide the user 
with sufficient privileges to tamper with unauthorized resources on the system 
(e.g., critical files, directories, etc.)."""

    if not fmt:
        print>>fo
        print>>fo, "RULE 0702"
        print>>fo, "---------"

    id      = 0xFFFFFFFF  # do a wildcard lookup
    context = 0
    cont = True
    while (cont):
        try:
            s, idn, idv, ida, context = starlet.idtoasc(id, context)
            if (idv < 0x80000000L):
                continue
            if idn in ('BATCH', 'DIALUP', 'INTERACTIVE', 'LOCAL', 'REMOTE',
                       'NETWORK', 'DECWINDOWS'):
                continue
            if idn.split('$')[0] in ('NET', 'SYS', 'SECSRV', 'VMS'):
                continue 
            try:
                s, holder, attrib = starlet.find_holder(idv)
            except VMSError, e:
                if e.errno != ssdef.SS__NOSUCHID:
                    raise e
                idvs = "%%X%X" % idv
                if fmt:
                    print>>fo, '07022', "id = %s idn = %s" % (idvs, idn)
                else:
                    print>>fo, "id = %s idn = %s" % (idvs, idn)
        except VMSError, e:
            if e.errno != ssdef.SS__NOSUCHID:
                raise e
            cont = False

    starlet.finish_rdb(context)

@level_rule(2)
def rule0703(fo, fmt):
    """ This may allow vulnerabilities to exist in these tables making them 
susceptible to unwanted changes.  The owner of a table can freely add, modify, 
and delete entries in the table."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0703'
        print>>fo, '========='

    it = [itemList.itemList(code=ossdef.OSS__OWNER, dtype=itemList.il_unsignedLong),]

    own = starlet.get_security (objnam='LNM$SYSTEM_TABLE', clsnam='LOGICAL_NAME_TABLE',
                              itmlst=it)[1][ossdef.OSS__OWNER]

    high_word = int(own / 65536)
    low_word  = int(own - (high_word *65536))
    own = "[%o,%o]" % (high_word, low_word)
    if own != '[1,4]':
        if fmt:
            print>>fo, '07032', own
        else:
            print>>fo, own

@level_rule(2)
def rule0704(fo, fmt):
    """ This allows any user to potentially change these system tables by 
removing or adding elements to them.  Such actions may allow a user to 
circumvent certain system security measures or override safeguards."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0704'
        print>>fo, '========='

    it = [itemList.itemList(code=ossdef.OSS__PROTECTION, dtype=itemList.il_unsignedLong),]
    prot = starlet.get_security (objnam='LNM$SYSTEM_TABLE', clsnam='LOGICAL_NAME_TABLE',
                                 itmlst=it)[1][ossdef.OSS__PROTECTION]

    if  not ((prot & 0x8000) and (prot & 0x4000) and (prot & 0x2000)):
        if fmt:
            print>>fo, '07042 LNM$SYSTEM_TABLE'
        else:
            print>>fo, 'LNM$SYSTEM_TABLE', lib.format_sogw_prot (prot)[1]

@level_rule(2)
def rule0705(fo, fmt):
    """ This allows any member of the owners group to potentially change these 
system tables by removing or adding elements to them.  Such actions may allow 
a user to circumvent certain system security measures or override safeguards."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0705'
        print>>fo, '========='

    it = [itemList.itemList(code=ossdef.OSS__PROTECTION, dtype=itemList.il_unsignedLong),]
    prot = starlet.get_security (objnam='LNM$SYSTEM_TABLE', clsnam='LOGICAL_NAME_TABLE',
                                 itmlst=it)[1][ossdef.OSS__PROTECTION]

    if  not ((prot & 0x800) and (prot & 0x400) and (prot & 0x200)):
        if fmt:
            print>>fo, '07052 LNM$SYSTEM_TABLE'
        else:
            print>>fo, 'LNM$SYSTEM_TABLE', lib.format_sogw_prot (prot)[1]

@level_rule(2)
def rule0706(fo, fmt):
    """ This may allow unauthorized access to users which have the associated 
identifiers granted to them."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0706'
        print>>fo, '========='

    it = [itemList.itemList(code=ossdef.OSS__ACL_LENGTH, dtype=itemList.il_unsignedLong),]
    acllen = starlet.get_security (objnam='LNM$SYSTEM_TABLE', clsnam='LOGICAL_NAME_TABLE',
                                   itmlst=it)[1][ossdef.OSS__ACL_LENGTH]

    if (acllen != 0):
        if fmt:
            print>>fo, '07062 LNM$SYSTEM_TABLE'
        else:
            print>>fo, get_security.get_security('LNM$SYSTEM_TABLE', clsnam='LOGICAL_NAME_TABLE')

@level_rule(1)
def rule0707(fo, fmt):
    """ This could result in unauthorized access to the operating system files, 
and compromise system integrity."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0707'
        print>>fo, '========='

    it = [itemList.itemList(code=ossdef.OSS__OWNER, dtype=itemList.il_unsignedLong),]

    ident = starlet.get_security (objnam='SYS$SYSDEVICE:', clsnam='DEVICE',
                                  itmlst=it)[1][ossdef.OSS__OWNER]

    high_word = int(ident / 65536)
    low_word  = int(ident - (high_word *65536))
    own = "[%o,%o]" % (high_word, low_word)
    if (own != '[1,4]') and (own != '[1,1]'):
        if fmt:
            print>>fo, '07071 SYS$SYSDEVICE:'
        else:
            print>>fo, 'SYS$SYSDEVICE:', own

@level_rule(2)
def rule0708(fo, fmt):
    """ A system shared disk should typically be owned by the SYSTEM account.  
A privately mounted disk should be owned by a valid user in the SYSUAF."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0708'
        print>>fo, '========='

    it = [itemList.itemList(code=ossdef.OSS__OWNER, dtype=itemList.il_unsignedLong),]

    ident = starlet.get_security (objnam='SYS$SYSDEVICE:', clsnam='DEVICE',
                                  itmlst=it)[1][ossdef.OSS__OWNER]
    high_word = int(ident / 65536)
    low_word  = int(ident - (high_word *65536))
    own = "[%o,%o]" % (high_word, low_word)

    try:
        r = starlet.idtoasc(ident)
        g, m =user_exists.user_exists(r[1])
        if g is None:
            if fmt:
                print>>fo, '07082 SYS$SYSDEVICE:', r[1]
            else:
                print>>fo, 'SYS$SYSDEVICE:', r[1], own 
    except VMSError, e:
        if e == ssdef.SS__NOSUCHID:
            if fmt:
                print>>fo, '07082 SYS$SYSDEVICE:'
            else:
                print>>fo, 'SYS$SYSDEVICE:', own
        else:
            raise e

@level_rule(2)
def rule0709(fo, fmt):
    """ Incorrect disk protection can yield undesirable access of sensitive 
data or files to unauthorized users."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0709'
        print>>fo, '========='

    it = [itemList.itemList(code=ossdef.OSS__PROTECTION, dtype=itemList.il_unsignedLong),]

    prot = starlet.get_security (objnam='SYS$SYSDEVICE:', clsnam='DEVICE',
                              itmlst=it)[1][ossdef.OSS__PROTECTION]

    if not (prot & 0xFE00):
        if fmt:
            print>>fo, '07092 SYS$SYSDEVICE:'
        else:
            print>>fo, 'SYS$SYSDEVICE:', lib.format_sogw_prot (prot)[1]

@level_rule(2)
def rule0710(fo, fmt):
    """ Highwater marking is a feature of OpenVMS which protects users from 
reading files beyond what was written to them.  This also is termed disk 
scavenging."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0710'
        print>>fo, '========='

    for device in DeviceScan('*', devclass=dcdef.DC__DISK):
        if not (lib.getdvi (dvidef.DVI__MNT, device_name=device)[1]):
            continue
        if lib.getdvi (dvidef.DVI__SHDW_MEMBER, device_name=device)[1]:
            continue
        if lib.getdvi (dvidef.DVI__NOHIGHWATER, device_name=device)[1]:
            if fmt:
                print>>fo, '07102', device
            else:
                print>>fo, device, 'NOHIGHWATER'

@level_rule(2)
def rule0711(fo, fmt):
    """ This feature writes a pattern over the disk space occupied by a file 
when the file is deleted.  This prevents users from scavenging old data from 
the disk which might otherwise be considered reserved only for authorized 
users."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0711'
        print>>fo, '========='

    for device in DeviceScan('*', devclass=dcdef.DC__DISK):
        if not (lib.getdvi (dvidef.DVI__MNT, device_name=device)[1]):
            continue
        if lib.getdvi (dvidef.DVI__SHDW_MEMBER, device_name=device)[1]:
            continue
        if not lib.getdvi (dvidef.DVI__ERASE_ON_DELETE, device_name=device)[1]:
            if fmt:
                print>>fo, '07112', device
            else:
                print>>fo, device, 'NO ERASE_ON_DELETE'

@level_rule(2)
def rule0712(fo, fmt):
    """ This allows the owner full access to the volume, which can present a 
security risk."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0712'
        print>>fo, '========='

    it = [itemList.itemList(code=ossdef.OSS__OWNER, dtype=itemList.il_unsignedLong),]

    ident = starlet.get_security (objnam='SYS$SYSDEVICE:', clsnam='DEVICE',
                                  itmlst=it)[1][ossdef.OSS__OWNER]
    high_word = int(ident / 65536)
    low_word  = int(ident - (high_word *65536))
    own = "[%o,%o]" % (high_word, low_word)

    if (own != '[1,4]') and (own != '[1,1]'):
        if fmt:
            print>>fo, '07122 Device SYS$SYSDEVICE:'
        else:
            print>>fo, 'Device SYS$SYSDEVICE:', own 

@level_rule(2)
def rule0713(fo, fmt):
    """ Owners of a volume should be defined in the SYSUAF.  This could 
present a security problem, if a user has privileges to define the owner into 
the SYSUAF and then login to that account.  This would permit the user full 
privileges on the volume."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0713'
        print>>fo, '========='

    it = [itemList.itemList(code=ossdef.OSS__OWNER, dtype=itemList.il_unsignedLong),]

    ident = starlet.get_security (objnam='SYS$SYSDEVICE:', clsnam='VOLUME',
                                  itmlst=it)[1][ossdef.OSS__OWNER]
    high_word = int(ident / 65536)
    low_word  = int(ident - (high_word *65536))
    own = "[%o,%o]" % (high_word, low_word)
    if own != '[1,1]':
        try:
            r = starlet.idtoasc(ident)
            g, m =user_exists.user_exists(r[1])
            if g is None:
                if fmt:
                    print>>fo, '07132 Volume SYS$SYSDEVICE:'
                else:
                    print>>fo, 'SYS$SYSDEVICE:', r[1], own 
        except VMSError, e:
            if e == ssdef.SS__NOSUCHID:
                if fmt:
                    print>>fo, '07132 Volume SYS$SYSDEVICE:'
                else:
                    print>>fo, 'SYS$SYSDEVICE:', own
            else:
                raise e

@level_rule(2)
def rule0714(fo, fmt):
    """ Incorrect volume protections can prevent users from accessing to the 
disks.  By default, all users should be granted RWCD access to the volume."""

    if not fmt:
        print>>fo
        print>>fo, 'RULE 0714'
        print>>fo, '========='

    it = [itemList.itemList(code=ossdef.OSS__PROTECTION, dtype=itemList.il_unsignedLong),]

    prot = starlet.get_security (objnam='SYS$SYSDEVICE:', clsnam='VOLUME',
                                  itmlst=it)[1][ossdef.OSS__PROTECTION]
    accnam = lib.get_accnam('VOLUME')[1]
    pvw = lib.format_sogw_prot (prot,access_names=accnam)[1].split(',')[3]
    if pvw[8:] != 'RWCD':
        if fmt:
            print >>fo, '07142SYS$SYSDEVICE:', pvw
        else:
            print >>fo, 'SYS$SYSDEVICE: bad World protection', pvw


if __name__ == '__main__':
    import sys
    fo = open(sys.argv[1], 'w') if len(sys.argv) > 1 else sys.stdout
    rule0701(fo, len(sys.argv) > 2)
    rule0702(fo, len(sys.argv) > 2)
    rule0703(fo, len(sys.argv) > 2)
    rule0704(fo, len(sys.argv) > 2)
    rule0705(fo, len(sys.argv) > 2)
    rule0706(fo, len(sys.argv) > 2)
    rule0707(fo, len(sys.argv) > 2)
    rule0708(fo, len(sys.argv) > 2)
    rule0709(fo, len(sys.argv) > 2)
    rule0710(fo, len(sys.argv) > 2)
    rule0711(fo, len(sys.argv) > 2)
    rule0712(fo, len(sys.argv) > 2)
    rule0713(fo, len(sys.argv) > 2)
    rule0714(fo, len(sys.argv) > 2)
