/*****************************************************************************/
/*
                                   pyRTE.c

A WASD Run-Time Envrionment (persistent engine) for Python.

The basics of this RTE processes with a CGIplus request cycle (as do ALL RTEs). 
The Python scripts do not need to be aware they are executing in a persistent
engine.  CGI variables are available from the os.environ and POSTed content
from the <stdin> stream so all out-of-the-box Python CGI scripts should work
without modification.

The byte-code used is either loaded from pre-compiled file content (.PYO or
.PYC coresident with the .PY) or compiled dynamically from a source file (.PY).
It maintains a cache of this byte-code for a number of scripts (determined by
compilation constant and logical name value).  Benchmarks on the elementary
"hello world" test script indicates an increase in throughput of 30% compared
to when the cache is disabled.

pyRTE also supports Python scripts explicitly implementing CGIplus themselves. 
All that is required of the script is a call to wasd.cgiplus_begin() to
synchonise the receipt of a new request.  All other CGIplus functionality is
handled transparently by PYRTE.C.  CGI variables are present in the os.environ
and POSTed content is as usual available form <stdin>.

A minimal example CGIplus script:

  import wasd
  while wasd.cgiplus_begin():
     print 'Content-type: text/html'
     print ''
     print '<h1><i>hello world!</i></h1>'

The cgiplus_begin() method accepts zero, one or two positional parameters.

1) The first is a boolean (True or False) that if true allows a standard CGI
script to call this once before returning false at the second call.  This
eliminates the need for explicit CGI and CGIplus paths.

2) The second parameter is also a boolean (True or False).  If true it gets
the script code modification time to check for changes.  If modified it returns
false causing the CGIplus script to exit gracefully (and consequently have the
new code run-up by the server).

NOTE: This RTE needs to differentiate between operating in CGIplus mode, where
the RTE activated scripts that are unaware they are operating in a persistent
engine environment, and when an explicitly CGIplus script is using the RTE as a
Python engine.  Hence two flags in the code; IsCgiPlusMode which indicates it's
not standard CGI, just an RTE, and IsCgiPlusScript which indicates it's been
activated as a CGIplus script.


LATENCIES
---------
Establishing a Python interpreter is quite expensive in terms of CPU cycles and
latency.  When pyRTE creates a new interpreter for each request it means that
script name-spaces are completely isolated.  On my test-bench PWS500 this adds
approximately 250mS to each request.  pyRTE attempts to ameliorate the effects
of this overhead by creating the interpreter prior to accepting the next
request.  For back-to-back requests this gains nothing but where there is at
least 250mS between requests (minimum, and for example) reduces response time
latency by that amount.  Hence the simple "hello world" test example for
back-to-back requests (generated using ApacheBench or WASD-bench) each is
reported by HTTPDMON as having a duration of approximately 280mS, while
one-at-a-time using a browser each is reported at between 10 and 15 mS.

BY DEFAULT, pyRTE takes AN AGGRESSIVE POSITION ON REDUCING LATENCY.  This
leaves the interpreter intact for each unique script.  The standard CGI "hello
world" has it's performance pushed from 2.5 requests/second to 80/S (on my
PWS500au test-bench), an improvement of some 35x!!  A unique main dictionary
is created for each request but there is some potential for any 'fiddles'
under-the-hood (e.g. the sort of thing cgitb does to exception handling) may
cause interactions between request invocations.  As the same interpreter is
only used for the same script (based on a unique script file-system name) it is
suggested that these may be few-and-far-between, and perhaps mostly benign. 
Anyway, this mode of interpreter reuse can be disabled on a per-RTE basis using
the PYRTE_NOREUSE_INTERPRETER logical name, or using the per-script mapping
rule 'script=param="PYRTE=/NOREUSE"', or by the Python script using the
wasd.reuse_interpreter(False) function sometime during execution.

CGIplus scripting also shines in the latency department.  With "hello world"
running as CGIplus back-to-back latency drops to approximately 6mS!!  This
translates to something like 150 requests/second rather than 3.5/S.  Nice! 
Surely with Python's automatic garbage collection and a little care persistent
CGIplus applications in Python should be eminently doable.


WASD PYTHON MODULE
------------------
The RTE implicitly makes available a module that provides additional
functionality for CGIplus implementations and other purposes.  The module still
needs to be imported into a Python application:

  import wasd

Python functions provided by this module:

  wasd.cgiplus_begin()

    This function is used to synchronise an explicitly CGIplus script
    (see example above).  It accepts zero, one or two arguments.

  wasd.cgiplus_end()

    Can be used to explicitly to end a CGIplus request.  Normally that
    functionality would be handled implicitly using wasd.cgiplus_begin().

  wasd.cgivar()

    Normally all CGI variables may be accessed via os.environ but this
    function can be used to obtain the same data (and some variables
    that don't make it into os.environ (e.g. FORM_..).

  wasd.reuse_interpreter()

    See discussion of 'latencies' above.  Takes one arugment; True to reuse
    the current interpreter next request, or False to use a fresh interpreter.

  wasd.rte_cache_entry()

    Returns a string containing details of the particular cache entry as a
    series of comma-separated-values (CSV) in the same order as the data
    structure.  Multiple calls step through each cache element until None
    is returned.  Note that this function will not succeed without logical
    name PYRTE_CACHE_ENTRY being defined.

  wasd.rte_id()

    Returns a string containing the RTE software identification.

  wasd.stat_timer()

    Returns a string containing some LIB$STAT_TIMER() data.

  wasd.usage_code()

    Returns an integer representing the number of times the cached byte-code
    for this particular script has been (re)used.

  wasd.usage_interpreter()

    Returns an integer representing the number of times this (sub)interpreter
    has been used (if enabled).

  wasd.usage_rte()

    Returns an integer representing the number of times this RTE has been used.


REFERENCES
----------
Python/C API Reference Manual, Release 2.4.3
Extending and Embedding the Python Interpreter, Release 2.4.3
Lots of Googling with many and varied query strings.


PYRTE COPYRIGHT
---------------
Copyright (C) 2007 Mark G.Daniel
This package comes with ABSOLUTELY NO WARRANTY.
This is free software, and you are welcome to redistribute it
under the conditions of the GNU GENERAL PUBLIC LICENSE, version 2.

  http://www.gnu.org/licenses/gpl.txt


PYTHON COPYRIGHT
----------------
Python is subject to it's own PSF license.

  http://www.python.org/psf/license.html


MAPPING RULES
-------------
For standard (non-CGIplus) Python scripts ...

To make any in-production .PY and associated .PYO or .PYC file in
CGI-BIN:[000000] available as an executable Python script:

  exec /cgi-bin/**.py* (cgi-bin:[000000]pyrte.exe)/cgi-bin/*.py* \
       script=syntax=unix script=query=none map=once ods=5

For Python scripts implementing CGIplus themselves ...

Requires the configuration directive:

  [DclRunTime]
  .PY $CGI-BIN:[000000]PYRTE.EXE

And the mapping:

  exec+ /cgiplus-bin/**.py* /cgi-bin/*.py*
        script=syntax=unix script=query=none map=once ods=5

Of course this may be adapted to Python scripts in any location.

For PYRTE development purposes this rule directly makes available the test
Python scripts in the source directory as if they were in /cgi-bin/:

  exec /cgi-bin/pyrte_**.py* \
       (cgi-bin:[000000]pyrte.exe)/ht_root/src/pyrte/pyrte_*.py* \
       script=query=none map=once ods=5

  exec+ /cgiplus-bin/pyrte_**.py* /ht_root/src/pyrte/pyrte_*.py*
        script=syntax=unix script=query=none map=once ods=5


SCRIPT PARAMETERS
-----------------
These parameters can control aspects of RTE behaviour on a per-path (and
therefore per-script) basis using mapping rules.

  script=param=PYRTE=/REUSE      reuse this Python interpreter
  script=param=PYRTE=/NOREUSE    do not reuse this Python interpreter
  script=param=PYRTE=/NOSTREAM   do not issue the stream-mode header


LOGICAL NAMES
-------------
PYRTE$DBUG         enables all if(Debug) output statements;
                     if some unique part of the script debugs just that script
                     if defined to something like 'c' or '/' or '-' debugs all
                     (this script-name check is CASE-SENSITIVE)
PYRTE_BASIC_RTE    full interpreter cycle request processing (for comparison)
PYRTE_CACHE_ENTRY  enables function RtePyMethRteCache()
PYRTE_CACHE_MAX    integer size of code cache (zero to compare uncached)
PYRTE_NOREUSE_INTERPRETER  do not reuse interpreter for each script instance
PYRTE_USAGE_LIMIT  integer number of requests before voluntary exit


BUILD DETAILS
-------------
$ @BUILD_PYRTE BUILD  !compile+link
$ @BUILD_PYRTE LINK   !link-only
$ COPY HT_EXE:PYRTE.EXE CGI_EXE:


VERSION HISTORY (update SOFTWAREVN as well!)
---------------
22-APR-2007  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#define SOFTWAREVN "1.0.0"
#define SOFTWARENM "PYRTE"
#ifdef __ALPHA
#  define SOFTWAREID SOFTWARENM " AXP-" SOFTWAREVN
#endif
#ifdef __ia64
#  define SOFTWAREID SOFTWARENM " IA64-" SOFTWAREVN
#endif
#ifdef __VAX
#  define SOFTWAREID SOFTWARENM " VAX-" SOFTWAREVN
#endif

/************/
/* includes */
/************/

/* Python-specific includes (must come first!) */

#include <python.h>

/* standard C header files */

#include <ctype.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unixlib.h>
#include <unixio.h>

/* VMS-specific header files */

#include <descrip.h>
#include <lib$routines.h>
#include <lnmdef.h>
#include <ssdef.h>
#include <starlet.h>
#include <syidef.h>

/**********/
/* macros */
/**********/

#ifndef BUILD_DATETIME
#  define BUILD_DATETIME "(undefined)"
#endif

#define FI_LI __FILE__, __LINE__

/******************/
/* global storage */
/******************/

int  CgiPlusPythonEnd,
     Debug,
     GlobalDebug,
     IsCgiPlusMode,
     IsCgiPlusScript,
     NoStreamMode,
     ReuseInterpreter,
     UsageCount,
     UsageLimit;

char  *CgiPlusEofPtr,
      *CgiPlusEotPtr,
      *CgiPlusEscPtr;

char  CgiPlusEof [64],
      CgiPlusEot [64],
      CgiPlusEsc [64];

char  ErrorCgiVarEnv [] = "Error initialising CGI environment.",
      ErrorOverflow [] = "Buffer overflow!",
      ErrorPython [] = "Error initialising Python environment.";


/****************************/
/* python extension methods */
/****************************/

static PyObject* RtePyMethCgiPlusBegin (PyObject*, PyObject*);

PyDoc_STRVAR (rte_cgiplus_begin__doc__,
"For a Python script implementing the CGIplus protocol itself this method \
synchronises against the receipt of a new request from the server. It handles \
all of the stream flushing and sentinal output required of WASD CGIplus.");

static PyObject* RtePyMethCgiPlusEnd (PyObject*, PyObject*);

PyDoc_STRVAR (rte_cgiplus_end__doc__,
"For a Python script implementing the CGIplus protocol itself this method \
provides all of the flushing and sentinal output required at the end of \
request process. This functionality is implicitly provided by the \
cgiplus_begin() method and so generally is not explicitly required.");

static PyObject* RtePyMethGetVar (PyObject*, PyObject*);

PyDoc_STRVAR (rte_getvar__doc__,
"Returns a string containing the value of the CGI variable specified by name \
in the function argument. Generally CGI variables are accessed from \
os.environ in the same manner as other environment variables.");

static PyObject* RtePyMethReuseInterpreter ();

PyDoc_STRVAR (rte_reuse_interpreter__doc__,
"Enables (True) or disables (False) interpreter reuse for the next request \
received by pyRTE.");

static PyObject* RtePyMethRteCacheEntry (PyObject*, PyObject*);

PyDoc_STRVAR (rte_cache_entry__doc__,
"Returns a string containing details of a pyRTE cache entry as a series of \
comman-separated values (CSV) in the same order as the data structure. \
Multiple calls step through each cache element until None is returned. \
This function will not succeed unless logical name PYRTE_CACHE_ENTRY is \
defined.");

static PyObject* RtePyMethRteId ();

PyDoc_STRVAR (rte_id__doc__,
"Returns a string containing the pyRTE version and build date/time.");

static PyObject* RtePyMethStatTimer ();

PyDoc_STRVAR (rte_stat_timer__doc__,
"Returns a string containing LIB$STAT_TIMER() statistics for the request.");

static PyObject* RtePyMethUsageCode ();

PyDoc_STRVAR (rte_usage_code__doc__,
"Returns an integer value for how many times the cached byte-code \
has been used.");

static PyObject* RtePyMethUsageInterpreter ();

PyDoc_STRVAR (rte_usage_interpreter__doc__,
"Returns an integer value for how many times the current Python \
(sub)interpreter has been used.");

static PyObject* RtePyMethUsageRte ();

PyDoc_STRVAR (rte_usage_rte__doc__,
"Returns an integer value for how many times the RTE has been used.");

/* Jean-Franois Pironne's original '_wasd' module functions */ 
static PyObject* wasd_cgi_init ();
static PyObject* wasd_isCgiPlus ();
static PyObject* wasd_cgi_eof ();
static PyObject* wasd_cgi_info ();
static PyObject* wasd_cgi_read ();

static struct PyMethodDef RtePyMethods [] =
{
   { "cgiplus_begin", (PyCFunction)RtePyMethCgiPlusBegin, METH_VARARGS,
                       rte_cgiplus_begin__doc__ },
   { "cgiplus_end", (PyCFunction)RtePyMethCgiPlusEnd, METH_VARARGS,
                     rte_cgiplus_end__doc__ },
   { "getvar", (PyCFunction)RtePyMethGetVar, METH_VARARGS,
                rte_getvar__doc__ },
   { "reuse_interpreter", (PyCFunction)RtePyMethReuseInterpreter, METH_VARARGS,
                           rte_reuse_interpreter__doc__ },
   { "rte_cache_entry", (PyCFunction)RtePyMethRteCacheEntry, METH_VARARGS,
                        rte_cache_entry__doc__ },
   { "rte_id", (PyCFunction)RtePyMethRteId, METH_VARARGS,
                rte_id__doc__ },
   { "stat_timer", (PyCFunction)RtePyMethStatTimer, METH_NOARGS,
                    rte_stat_timer__doc__ },
   { "usage_code", (PyCFunction)RtePyMethUsageCode, METH_NOARGS,
                    rte_usage_code__doc__ },
   { "usage_interpreter", (PyCFunction)RtePyMethUsageInterpreter, METH_NOARGS,
                           rte_usage_interpreter__doc__ },
   { "usage_rte", (PyCFunction)RtePyMethUsageRte, METH_NOARGS,
                   rte_usage_rte__doc__ },

   /* JFP's original '_wasd' module functions */ 
   { "cgi_init", (PyCFunction)wasd_cgi_init, 0,
                 "initialise CGIplus environment" },
   { "isCgiPlus", (PyCFunction)wasd_isCgiPlus, 0,
                  "is it activated in CGIplus mode?" },
   { "cgi_eof", (PyCFunction)wasd_cgi_eof, 0,
                "concludes the current CGIplus request" },
   { "cgi_info", (PyCFunction)wasd_cgi_info, METH_VARARGS,
                 "return a CGI variable" },
   { "read", (PyCFunction)wasd_cgi_read, METH_VARARGS,
             "read the entire POSTed body" },

   { NULL, NULL, 0, NULL }
};

/*********************/
/* global code cache */
/*********************/

/* somewhat arbitrary */
#define CACHE_MAX 32

int  CodeCacheCount = 0,
     CodeCacheMax = CACHE_MAX;

struct CodeCacheStruct
{
   int  CodeUsageCount,
        InterpreterUsageCount;
   unsigned long  LastUsedSecs,
                  ScriptMtimeSecs;
   char *FileNamePtr,
        *FileNameByteCodePtr;
   PyThreadState  *pInterpState;
   PyCodeObject  *pByteCode;
}
CodeCache [CACHE_MAX];

struct CodeCacheStruct  *CodeCachePtr;

/**************/
/* prototypes */
/**************/

void at_Exit ();
int ByteCodeUpToDate (struct CodeCacheStruct*);
char* CgiVar (char*);
char* CgiVarDclSymbol (char*);
void LoadByteCode (char*, char*, struct CodeCacheStruct*);
int ModuleCgiibLoaded ();
void ProcessCGI (int, char**);
void ProcessCGIplus (int, char**);
void ProcessBasicRte ();
void ProcessCachingRte ();
void ReportError (int, int, int, int, char*, ...);
int RemoveOsEnvCgiVar ();
void RteScriptParam ();
int SetCrtlFeature (char*, int);
int SetupOsEnvCgiVar ();
char* TrnLnm (char*, char*);
char* StatTimer (int);

/* not in an include file for some reason (adapted from node.h)*/
PyAPI_FUNC(void) PyNode_Free (struct _node*);
PyAPI_FUNC(PyObject*) PyMarshal_ReadObjectFromFile(FILE*);
unsigned long PyMarshal_ReadLongFromFile(FILE*);

/*****************************************************************************/
/*
*/

main (int argc, char *argv[])
       
{
   char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   /* this is almost a 'bit too clever' :-) */
   if ((cptr = TrnLnm ("PYRTE$DBUG", NULL)) != NULL)
      if (sptr = CgiVar ("SCRIPT_NAME"))
         if (strstr (sptr, cptr))
            if (GlobalDebug = Debug = 1)
               fprintf (stdout, "Content-Type: text/plain\n\n");

   /* if it doesn't look like CGI environment then forget it */
   if (!TrnLnm ("HTTP$INPUT", NULL)) exit (SS$_ABORT);

   /* VMS Python is only supported on V7.3 and later - and so are these! */
   /* All others DECC$ features are defined by the Python shareable image */
/**   SetCrtlFeature ("DECC$EFS_CASE_SPECIAL", 1); **/
/**   SetCrtlFeature ("DECC$FILE_SHARING", 1); **/

   if (!Debug)
   {
      /* binary mode to eliminate carriage-control */
      if (!(stdin = freopen ("HTTP$INPUT:", "r", stdin,
                             "ctx=bin")))
         exit (vaxc$errno);
      if (!(stdout = freopen ("SYS$OUTPUT:", "w", stdout,
                              "ctx=bin", "ctx=xplct")))
         exit (vaxc$errno);
      if (!(stdout = freopen ("SYS$ERROR:", "w", stderr,
                              "ctx=bin", "ctx=xplct")))
         exit (vaxc$errno);
   }

   cptr = TrnLnm ("PYRTE_USAGE_LIMIT", NULL);
   if (cptr && isdigit(*cptr)) UsageLimit = atoi(cptr);

   /* need differentiate between RTE/CGIplus and CGIplus script */
   IsCgiPlusMode = (TrnLnm ("CGIPLUSEOF", NULL) != NULL);
   CgiPlusEofPtr = TrnLnm ("CGIPLUSEOF", CgiPlusEof);
   CgiPlusEotPtr = TrnLnm ("CGIPLUSEOT", CgiPlusEot);
   CgiPlusEscPtr = TrnLnm ("CGIPLUSESC", CgiPlusEsc);

   atexit (&at_Exit);

   if (IsCgiPlusMode)
   {
      /* for pyRTE an RTE will only have one parameter, CGIplus two! */
      if (argc == 1)
      {
         /*******/
         /* RTE */
         /*******/

         if (TrnLnm ("PYRTE_BASIC_RTE", NULL))
            ProcessBasicRte ();
         else
            ProcessCachingRte ();
      }
      else
      {
         /***********/
         /* CGIplus */
         /***********/

         /* supply the command-line script file name */
         ProcessCGIplus (argc, argv);
      }
   }
   else
   {
      /*******/
      /* CGI */
      /*******/

      ProcessCGI (argc, argv);
   }

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Just flush() needed due to the "ctx=xplct" in the freopen().
*/

void at_Exit ()
       
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "at_Exit()\n");

   fflush (stdout);
   fflush (stderr);
}

/*****************************************************************************/
/*
Process with a full, standard CGI request cycle.
For base-line comparison purposes only.
*/

void ProcessCGI
(
int argc,
char **argv
)
{
   int  status;
   char  *cptr, *sptr;
   FILE  *fp;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessCGI()\n");

   Py_Initialize();
   Py_InitModule ("wasd", RtePyMethods);

   sptr = CgiVar ("SCRIPT_NAME");
   Py_SetProgramName (sptr);

   /* NEEDS WORK!! */
   PySys_SetArgv (argc, argv);

   cptr = CgiVar ("SCRIPT_FILENAME");
   fp = fopen (cptr, "r");
   if (fp)
   {
      PyRun_AnyFileExFlags (fp, cptr, 0, NULL);
      fclose (fp);
   }
   else
   {
      status = vaxc$errno;
      ReportError (__LINE__, 1, status, 0, "Error opening !AZ", sptr);
   }

   Py_Finalize();
}

/*****************************************************************************/
/*
For a Python script that implements CGIplus itself.
*/

void ProcessCGIplus
(
int argc,
char **argv
)
{
   int  status;
   char  *cptr, *sptr;
   struct CodeCacheStruct  *ccptr;
   PyObject  *pBuiltins,
             *pError,
             *pGlobalDict,
             *pMainDict,
             *pMainModule;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessCGIplus() %d\n", argc);

   /* code behaviours are different for an explicitly CGIplus script */
   IsCgiPlusScript = 1;

   Py_Initialize();

   /* just use the first element of the code cache */
   ccptr = &CodeCache[0];

   /* set the global pointer used by some functions */
   CodeCachePtr = ccptr;

   /* it will have bee used the once! */
   ccptr->InterpreterUsageCount++;

   Py_SetProgramName (argv[1]);

   /* first is RTE image, second CGIplus script, then optional parameters */
   PySys_SetArgv (argc-1, &argv[1]);

   LoadByteCode (argv[1], argv[1], ccptr);
   if (!ccptr->pByteCode) return;

   pGlobalDict = PyDict_New();
   if (!pGlobalDict)
   {
      ReportError (__LINE__, 1, 0, 1, NULL);
      return;
   }

   pMainModule = PyImport_AddModule ("__main__");
   if (!pMainModule)
   {
      ReportError (__LINE__, 1, 0, 0, ErrorPython);
      return;
   }

   pMainDict = PyModule_GetDict (pMainModule);
   if (!pMainDict)
   {
      ReportError (__LINE__, 1, 0, 0, ErrorPython);
      return;
   }

   pBuiltins = PyEval_GetBuiltins ();
   if (!pBuiltins)
   {
      ReportError (__LINE__, 1, 0, 0, ErrorPython);
      return;
   }

   if (PyDict_SetItemString (pBuiltins, "__persistdict__", pGlobalDict))
   {
      ReportError (__LINE__, 1, 0, 0, ErrorPython);
      return;
   }

   if (!Py_InitModule ("wasd", RtePyMethods))
   {
      ReportError (__LINE__, 1, 0, 1, NULL);
      return;
   }

   /* get the first request (which is already available and won't block) */
   CgiVar ("");

   if (!GlobalDebug && ((cptr = TrnLnm ("PYRTE$DBUG", NULL)) != NULL))
      if (sptr = CgiVar ("SCRIPT_NAME"))
         if (strstr (sptr, cptr))
            if (Debug = 1)
               fprintf (stdout, "Content-Type: text/plain\n\n");

   RteScriptParam ();

   if (!NoStreamMode) fprintf (stdout, "Script-Control: X-stream-mode\n");

   if (!SetupOsEnvCgiVar ()) return;

   PyErr_Clear ();

   PyEval_EvalCode (ccptr->pByteCode, pMainDict, pMainDict);
   if (pError = PyErr_Occurred())
   {
      if (!PyErr_GivenExceptionMatches (pError, PyExc_SystemExit))
      {
         ReportError (__LINE__, 0, 0, 1, NULL);
         free (ccptr->FileNamePtr);
         ccptr->FileNamePtr = NULL;
      }
   }

   fflush (stdout);
   fputs (CgiPlusEofPtr, stdout);
   fflush (stdout);

   if (!GlobalDebug) Debug = 0;

   Py_Finalize();
}

/*****************************************************************************/
/*
Process with a CGIplus request cycle but a full Python interpreter
initialization and rundown cycle as well.
For base-line comparison purposes only.
*/

void ProcessBasicRte ()
       
{
   int  status;
   char  *cptr, *sptr;
   char  *argv [2] = { NULL, NULL };
   FILE  *fp;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessBasicRte()\n");

   for (;;)
   {
      /* initialise this for special CGI-and-CGIplus script cases */
      RtePyMethCgiPlusBegin (NULL, NULL);

      /* try to get some of the work out of the way before next request */
      Py_Initialize();
      Py_InitModule ("wasd", RtePyMethods);

      /* block waiting for the first/next request */
      CgiVar ("");
      UsageCount++;

      if (!GlobalDebug && ((cptr = TrnLnm ("PYRTE$DBUG", NULL)) != NULL))
         if (sptr = CgiVar ("SCRIPT_NAME"))
            if (strstr (sptr, cptr))
               if (Debug = 1)
                  fprintf (stdout, "Content-Type: text/plain\n\n");

      RteScriptParam ();

      if (!NoStreamMode) fprintf (stdout, "Script-Control: X-stream-mode\n");

      cptr = CgiVar ("SCRIPT_FILENAME");

      Py_SetProgramName (cptr);

      /* fudge these */
      argv[0] = cptr;
      PySys_SetArgv (1, argv);

      fp = fopen (cptr, "r", "shr=get");
      if (fp)
      {
         PyRun_AnyFileExFlags (fp, cptr, 0, NULL);
         fclose (fp);
      }
      else
      {
         status = vaxc$errno;
         ReportError (__LINE__, 1, status, 0, "Error opening !AZ", sptr);
      }

      Py_Finalize();

      fflush (stdout);
      fputs (CgiPlusEofPtr, stdout);
      fflush (stdout);

      if (!GlobalDebug) Debug = 0;

      if (UsageLimit && UsageCount > UsageLimit) break;
   }
}

/*****************************************************************************/
/*
Process with a CGIplus request cycle.  Maintain a cache of the (compiled)
byte-code for a number of scripts (determined by compilation constant and
logical name value).  The byte-code used is either loaded from pre-compiled
file content (.PYO or .PYC) or compiled dynamically (and then cached) from a
source file (.PY).  Also, and based on RTE or script configuration, maintain a
cache of pre-initialized (and previously used) Python subinterpreters.
*/

void ProcessCachingRte ()
       
{
   int  idx, tidx, retval, status;
   unsigned long  mtime,
                  CurrentTimeSecs;
   char  *cptr, *sptr;
   char  *argv [2] = { NULL, NULL };
   struct CodeCacheStruct  *ccptr;
   PyObject  *pBuiltins,
             *pError,
             *pGlobalDict,
             *pMainDict,
             *pMainModule,
             *pResult;
   PyThreadState  *pStateBuffer,
                  *pNewInterpState = NULL;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessCachingRte()\n");

   cptr = TrnLnm ("PYRTE_CACHE_MAX", NULL);
   if (cptr && isdigit(*cptr)) CodeCacheMax = atoi(cptr);
   if (CodeCacheMax > CACHE_MAX) CodeCacheMax = CACHE_MAX;

   Py_Initialize();
   PyEval_InitThreads ();

   pGlobalDict = PyDict_New();
   if (!pGlobalDict)
   {
      ReportError (__LINE__, 1, 0, 1, NULL);
      return;
   }

   for (;;)
   {
      /********/
      /* wait */
      /********/

      if (!pNewInterpState)
      {
         /*
            Create a new interpreter *before* checking for the next request.
            This reduces response latency where requests are not back-to-back.
         */
         pStateBuffer = PyEval_SaveThread ();
         pNewInterpState = Py_NewInterpreter ();
         if (!pNewInterpState)
         {
            ReportError (__LINE__, 1, 0, 0, "Error initialising interpreter.");
            break;
         }
         PyThreadState_Swap (pStateBuffer);
      }

      /* initialise this for special CGI-and-CGIplus script cases */
      RtePyMethCgiPlusBegin (NULL, NULL);

      /* block waiting for the first/next request */
      CgiVar ("");
      UsageCount++;
      time (&CurrentTimeSecs);

      if (!GlobalDebug && ((cptr = TrnLnm ("PYRTE$DBUG", NULL)) != NULL))
         if (sptr = CgiVar ("SCRIPT_NAME"))
            if (strstr (sptr, cptr))
               if (Debug = 1)
                  fprintf (stdout, "Content-Type: text/plain\n\n");

      if (TrnLnm ("PYRTE_NOREUSE_INTERPRETER", NULL))
         ReuseInterpreter = 0;
      else
         ReuseInterpreter = 1;

      RteScriptParam ();

      if (!NoStreamMode) fprintf (stdout, "Script-Control: X-stream-mode\n");

      /***********/
      /* request */
      /***********/

      cptr = CgiVar ("SCRIPT_FILENAME");
      if (!cptr)
      {
         ReportError (__LINE__, 1, 0, 0, "Error getting SCRIPT_FILENAME.");
         break;
      }

      /* look to see if we have the script byte-code cached */
      ccptr = &CodeCache[0];
      for (idx = 0; idx < CodeCacheCount; idx++)
      {
         if (!ccptr->FileNamePtr) continue;
         if (Debug) fprintf (stdout, "|%s|%s|\n",
                             cptr, ccptr->FileNamePtr);
         if (!strcmp (cptr, ccptr->FileNamePtr)) break;
         ccptr++;
      }

      if (idx < CodeCacheCount)
      {
         /***************/
         /* found cache */
         /***************/

         if (!ByteCodeUpToDate (ccptr))
         {
            /* apparently not */
            free (ccptr->FileNamePtr);
            ccptr->FileNamePtr = NULL;
         }
      }
      else
      if (CodeCacheMax)
      {
         /*********************/
         /* new/refresh entry */
         /*********************/

         /* first look to see if there is an unused entry */
         for (tidx = 0; tidx < CodeCacheCount; tidx++)
            if (!CodeCache[tidx].FileNamePtr) break;
         if (tidx < CodeCacheCount)
            ccptr = &CodeCache[tidx];
         else
         if (idx < CodeCacheMax)
         {
            /* didn't find an unused entry */
            CodeCacheCount++;
            ccptr = &CodeCache[idx];
         }
         else
         {
            /* find the least recently used */
            for (idx = tidx = 0; idx < CodeCacheMax; idx++)
               if (CodeCache[idx].LastUsedSecs < CodeCache[tidx].LastUsedSecs)
                  tidx = idx;
            ccptr = &CodeCache[tidx];
            free (ccptr->FileNamePtr);
            ccptr->FileNamePtr = NULL;
         }
      }
      else
      {
         /* code caching is disabled */
         ccptr = &CodeCache[0];
         if (ccptr->FileNamePtr)
         {
            free (ccptr->FileNamePtr);
            ccptr->FileNamePtr = NULL;
         }
      }

      /* set the global pointer used by some functions */
      CodeCachePtr = ccptr;

      if (!ccptr->FileNamePtr)
      {
         /***************/
         /* load script */
         /***************/

         if (ccptr->pInterpState)
         {
            /* also lose any previously associated interpreter */
            pStateBuffer = PyThreadState_Swap (ccptr->pInterpState);
            Py_EndInterpreter (ccptr->pInterpState);
            PyThreadState_Swap (pStateBuffer);
            ccptr->pInterpState = NULL;
            ccptr->InterpreterUsageCount = 0;
         }

         sptr = CgiVar ("SCRIPT_NAME");
         if (!sptr)
         {
            ReportError (__LINE__, 1, 0, 0, "Error getting SCRIPT_NAME.");
            break;
         }

         LoadByteCode (cptr, sptr, ccptr);
      }

      if (ccptr->pByteCode)
      {
         /*********************/
         /* execute byte-code */
         /*********************/

         ccptr->CodeUsageCount++;
         ccptr->LastUsedSecs = CurrentTimeSecs;

         if (!ccptr->pInterpState)
         {
            /* associate an interpreter with the byte-code */
            ccptr->pInterpState = pNewInterpState;
            pNewInterpState = NULL;

            /* fudge the arguments */
            pStateBuffer = PyThreadState_Swap (ccptr->pInterpState);
            Py_SetProgramName (cptr);
            argv[0] = cptr;
            PySys_SetArgv (1, argv);
            PyThreadState_Swap (pStateBuffer);
         }
         ccptr->InterpreterUsageCount++;

         pStateBuffer = PyThreadState_Swap (ccptr->pInterpState);

         pMainModule = PyImport_AddModule ("__main__");
         if (!pMainModule)
         {
            ReportError (__LINE__, 1, 0, 0, ErrorPython);
            return;
         }

         pMainDict = PyModule_GetDict (pMainModule);
         if (!pMainDict)
         {
            ReportError (__LINE__, 1, 0, 0, ErrorPython);
            return;
         }

         /* just create as-clean-as-possible environment for the script */
         pMainDict = PyDict_Copy (pMainDict);
         if (!pMainDict)
         {
            ReportError (__LINE__, 1, 0, 0, ErrorPython);
            return;
         }

         pBuiltins = PyEval_GetBuiltins ();
         if (!pBuiltins)
         {
            ReportError (__LINE__, 1, 0, 0, ErrorPython);
            return;
         }

         if (PyDict_SetItemString (pBuiltins, "__persistdict__", pGlobalDict))
         {
            ReportError (__LINE__, 1, 0, 0, ErrorPython);
            return;
         }

         if (!Py_InitModule ("wasd", RtePyMethods))
         {
            ReportError (__LINE__, 1, 0, 1, NULL);
            return;
         }

         if (!SetupOsEnvCgiVar ()) return;

         PyErr_Clear ();

         pResult = PyEval_EvalCode (ccptr->pByteCode, pMainDict, pMainDict);
         if (pError = PyErr_Occurred())
         {
            if (!PyErr_GivenExceptionMatches (pError, PyExc_SystemExit))
            {
               ReportError (__LINE__, 0, 0, 1, NULL);
               free (ccptr->FileNamePtr);
               ccptr->FileNamePtr = NULL;
            }
         }

         if (!RemoveOsEnvCgiVar ()) return;

         PyThreadState_Swap (pStateBuffer);

         Py_XDECREF (pResult);
         Py_XDECREF (pMainDict);
      }
      else
      {
         /* no code - no cache! */
         free (ccptr->FileNamePtr);
         ccptr->FileNamePtr = NULL;
      }

      /******************/
      /* end of request */
      /******************/

      fflush (stdout);
      fputs (CgiPlusEofPtr, stdout);
      fflush (stdout);

      if (ccptr->pInterpState)
      {
         if (!ReuseInterpreter || !ccptr->FileNamePtr)
         {
            pStateBuffer = PyThreadState_Swap (ccptr->pInterpState);
            Py_EndInterpreter (ccptr->pInterpState);
            PyThreadState_Swap (pStateBuffer);
            ccptr->pInterpState = NULL;
            ccptr->InterpreterUsageCount = 0;
         }
      }

      if (!GlobalDebug) Debug = 0;

      if (UsageLimit && UsageCount > UsageLimit) break;
   }
}

/*****************************************************************************/
/*
Create the script byte-code by either loading the source code (.PY) and
compiling, or loading the pre-compiled byte code (.PYC) or optimised,
pre-compiled byte-code (.PYO).  Reports it's own errors.  Success is indicated
by '->pByteCode' containing a pointer to the code object, and failing by it
being NULL.
*/

void LoadByteCode
(
char *FileName,
char *ScriptName,
struct CodeCacheStruct *ccptr
)
{
   int  status,
        FileNameLength;
   unsigned long  magic, mtime;
   char  *cptr, *sptr, *zptr,
         *CharPtr;
   char  FileNameByteCode [256];
   FILE  *fp;
   stat_t  FstatBuffer;
   struct _node  *pNode;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "LoadByteCode() |%s|\n", FileName);

   ccptr->LastUsedSecs = ccptr->ScriptMtimeSecs = ccptr->CodeUsageCount = 0;

   if (ccptr->pByteCode)
   {
      /* remove reference to any cached code object */
      Py_DECREF (ccptr->pByteCode);
      ccptr->pByteCode = NULL;
   }

   if (ccptr->FileNamePtr)
   {
      free (ccptr->FileNamePtr);
      ccptr->FileNamePtr = NULL;
   }
   if (ccptr->FileNameByteCodePtr)
   {
      free (ccptr->FileNameByteCodePtr);
      ccptr->FileNameByteCodePtr = NULL;
   }

   zptr = (sptr = FileNameByteCode) + sizeof(FileNameByteCode);
   for (cptr = FileName; *cptr && sptr < zptr; *sptr++ = *cptr++);
   FileNameLength = cptr - FileName;
   CharPtr = sptr;
   /* first; try to open a '.pyo' (optimised byte-code) */
   *sptr++ = 'o';
   if (sptr > zptr)
   {
      ReportError (__LINE__, 1, 0, 0, ErrorOverflow);
      return;
   }
   *sptr = '\0';

   fp = NULL;
   for (;;)
   {
      if (Debug) fprintf (stdout, "|%s|\n", FileNameByteCode);
      /* note; opened in 'binary' mode */
      fp = fopen (FileNameByteCode, "rb", "shr=get");
      if (fp) break;
      /* third; try to open a '.py' (source) */
      if (*CharPtr == 'c') break;
      /* second; try to open a '.pyc' (compiled byte-code) */
      *CharPtr = 'c';
   }
   if (!fp)
   {
      /* well, it's certainly not pre-compiled! */
      FileNameByteCode[0] = '\0';
      if (Debug) fprintf (stdout, "|%s|\n", FileName);
      /* note; opened in 'record' mode */
      fp = fopen (FileName, "r", "ctx=rec", "shr=get");
      if (!fp)
      {
         status = vaxc$errno;
         ReportError (__LINE__, 1, status, 0, "Error opening !AZ", ScriptName);
         return;
      }
   }
   if (fstat (fileno(fp), &FstatBuffer) < 0)
   {
      status = vaxc$errno;
      fclose (fp);
      ReportError (__LINE__, 1, status, 0, "Error fstat() !AZ", ScriptName);
      return;
   }

   if (FileNameByteCode[0])
   {
      /* byte-code file; discard these first two longwords */
      magic = PyMarshal_ReadLongFromFile (fp);
      mtime = PyMarshal_ReadLongFromFile (fp);
      /* what's left is the byte-code */
      ccptr->pByteCode = (PyCodeObject*)PyMarshal_ReadObjectFromFile (fp);
      fclose (fp);
      if (!ccptr->pByteCode)
      {
         ReportError (__LINE__, 0, 0, 1, NULL);
         return;
      }
   }
   else
   {
      /* source code file */
      pNode = PyParser_SimpleParseFile (fp, ScriptName, Py_file_input);
      fclose (fp);
      if (!pNode)
      {
         ReportError (__LINE__, 1, 0, 1, NULL);
         return;
      }
      ccptr->pByteCode = (PyCodeObject*) PyNode_Compile (pNode, ScriptName);
      PyNode_Free (pNode);
      if (!ccptr->pByteCode)
      {
         ReportError (__LINE__, 1, 0, 1, NULL);
         return;
      }
   }

   /* successful load */
   ccptr->ScriptMtimeSecs = FstatBuffer.st_mtime;
   ccptr->FileNamePtr = calloc (1, FileNameLength+1);
   if (!ccptr->FileNamePtr) exit (vaxc$errno);
   strcpy (ccptr->FileNamePtr, FileName);
   if (FileNameByteCode[0])
   {
      ccptr->FileNameByteCodePtr = calloc (1, FileNameLength+2);
      if (!ccptr->FileNameByteCodePtr) exit (vaxc$errno);
      strcpy (ccptr->FileNameByteCodePtr, FileNameByteCode);
   }
   if (Debug) fprintf (stdout, "|%s|%s|\n", ccptr->FileNamePtr,
                       ccptr->FileNameByteCodePtr);
}

/*****************************************************************************/
/*
Check if this script (source or byte) code has been modified.
*/

int ByteCodeUpToDate (struct CodeCacheStruct *ccptr)

{
   int  retval;
   stat_t  StatBuffer;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ByteCodeUpToDate()\n");

   if (ccptr->FileNameByteCodePtr)
      retval = stat (ccptr->FileNameByteCodePtr, &StatBuffer);
   else
      retval = stat (ccptr->FileNamePtr, &StatBuffer);
   if (retval) return (0);
   /* if the script file is more recent than the cached byte-code */
   if (Debug) fprintf (stdout, "|%d|%d|\n",
                       StatBuffer.st_mtime, ccptr->ScriptMtimeSecs);
   if (StatBuffer.st_mtime > ccptr->ScriptMtimeSecs) return (0);
   return (1);
}

/*****************************************************************************/
/*
Populate the script's os.environ with CGI variables.
Some of the WASD-'standard' CGI variables are excluded.
*/

int SetupOsEnvCgiVar ()

{
   char  *cptr, *sptr;
   PyObject  *pEnvironDict,
             *pOsDict,
             *pOsModule,
             *pValue;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SetupOsEnvCgiVar()\n");

   pOsModule = PyImport_ImportModule ("os");
   if (!pOsModule)
   {
      ReportError (__LINE__, 1, 0, 0, ErrorCgiVarEnv);
      return (0);
   }

   pOsDict = PyModule_GetDict (pOsModule);
   if (!pOsDict)
   {
      ReportError (__LINE__, 1, 0, 0, ErrorCgiVarEnv);
      return (0);
   }

   pEnvironDict = PyDict_GetItemString (pOsDict, "environ");
   if (!pEnvironDict)
   {
      ReportError (__LINE__, 1, 0, 0, ErrorCgiVarEnv);
      return (0);
   }

   while (cptr = CgiVar ("*"))
   {
      if (Debug) fprintf (stdout, "|%s|\n", cptr);

      if ((toupper(*cptr) == 'F' &&
           !strncmp (cptr, "FORM_", 5)) ||
          (toupper(*cptr) == 'K' &&
           !strncmp (cptr, "KEY_", 4)) ||
          (toupper(*cptr) == 'P' &&
           !strcmp (cptr, "PYRTE")) ||
          (toupper(*cptr) == 'G' &&
           (!strncmp (cptr, "GATEWAY_EOF", 11) ||
            !strncmp (cptr, "GATEWAY_EOT", 11) ||
            !strncmp (cptr, "GATEWAY_ESC", 11) ||
            !strncmp (cptr, "GATEWAY_BG", 10))))
         continue;

      for (sptr = cptr; *sptr && *sptr != '='; sptr++);
      if (!*sptr) continue;
      *sptr = '\0';
      pValue = PyString_FromString (sptr+1);
      PyMapping_SetItemString (pEnvironDict, cptr, pValue);
      Py_DECREF (pValue);
      *sptr = '=';
   }

   Py_DECREF (pOsModule);

   return (1);
}

/*****************************************************************************/
/*
Remove currently defined CGI variables from the script's os.environ.  This is
used for CGIplus scripts to repopulate the CGI variable environment with the
next request's variables by explicitly removing each of the previous request's
variables at the end of it's processing.
*/

int RemoveOsEnvCgiVar ()

{
   int  retval;
   char  *cptr, *sptr;
   PyObject  *pEnvironDict,
             *pOsDict,
             *pOsModule;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RemoveOsEnvCgiVar()\n");

   pOsModule = PyImport_ImportModule ("os");
   if (!pOsModule)
   {
      ReportError (__LINE__, 1, 0, 0, ErrorCgiVarEnv);
      return (0);
   }

   pOsDict = PyModule_GetDict (pOsModule);
   if (!pOsModule)
   {
      ReportError (__LINE__, 1, 0, 0, ErrorCgiVarEnv);
      return (0);
   }

   pEnvironDict = PyDict_GetItemString (pOsDict, "environ");
   if (!pEnvironDict)
   {
      ReportError (__LINE__, 1, 0, 0, ErrorCgiVarEnv);
      return (0);
   }

   while (cptr = CgiVar ("*"))
   {
      if (Debug) fprintf (stdout, "|%s|\n", cptr);
      for (sptr = cptr; *sptr && *sptr != '='; sptr++);
      if (!*sptr) continue;
      *sptr = '\0';
      retval = PyMapping_DelItemString (pEnvironDict, cptr);
      if (Debug) fprintf (stdout, "PyMapping_DelItemString() %d\n", retval);
      *sptr = '=';
   }

   Py_DECREF (pOsModule);

   PyErr_Clear ();

   return (1);
}

/*****************************************************************************/
/*
Python method to return the string object value of the specified CGI variable
name.  An optional second Python function parameter allows a default object to
be returned if the CGI variable does not exist.  Uses all borrowed references!
*/

static PyObject* RtePyMethGetVar
(
PyObject *self,
PyObject *args
)
{
   char  *cptr, *sptr;
   PyObject  *pVarDefault = NULL,
             *pVarName = NULL;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RtePyMethGetVar()\n");

   if (!PyArg_ParseTuple (args, "S|O", &pVarName, &pVarDefault)) return (NULL);

   if (!(cptr = PyString_AsString (pVarName))) return (NULL);
   if (!(sptr = CgiVar (cptr)))
   {
      if (pVarDefault) return (pVarDefault);
      Py_INCREF(Py_None);
      return (Py_None);
   }

   return (Py_BuildValue ("s", sptr));
}

/*****************************************************************************/
/*
Python method to return the number of requests processed by this cached code.
*/

static PyObject* RtePyMethUsageCode ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RtePyMethUsageCode()\n");

   return (Py_BuildValue ("i", CodeCachePtr->CodeUsageCount));
}

/*****************************************************************************/
/*
Python method to return the number of requests processed by this cached code.
*/

static PyObject* RtePyMethUsageInterpreter ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RtePyMethUsageInterpreter()\n");

   return (Py_BuildValue ("i", CodeCachePtr->InterpreterUsageCount));
}

/*****************************************************************************/
/*
Python method to return the number of requests processed by the RTE.
*/

static PyObject* RtePyMethUsageRte ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RtePyMethUsageRte()\n");

   return (Py_BuildValue ("i", UsageCount));
}

/*****************************************************************************/
/*
Return a string object containing the LIB$STAT_TIMER statistics.
*/

static PyObject* RtePyMethStatTimer ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RtePyMethStatTimer()\n");

   return (Py_BuildValue ("s", StatTimer (TRUE)));
}

/*****************************************************************************/
/*
Python method to wait for the next CGIplus request.
If not CGIplus (i.e. standard CGI) the function returns immediately.
If a CGIplus request has been begun but not explicitly ended this function
provides the WASD script EOF sentinal.
It then blocks waiting for the next request.

It accepts zero, one or two parameters.  The first is a boolean (integer 0 or
1) that if True allows a standard CGI script to call this once before
returning False.  The second parameter is also a boolean (integer 0 or 1).  If
True it gets the script code modification time to check for changes.  If
modified it returns False.
*/

static PyObject* RtePyMethCgiPlusBegin
(
PyObject *self,
PyObject *args
)
{
   static int  CallCount = 0;

   int  AllowSingleCGI = 0,
        CheckCodeMtime = 0;
   char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (GlobalDebug) fprintf (stdout, "RtePyMethCgiPlusBegin()\n");

   if (!self && !args)
   {
      /* special case (not Python) reset the call count */
      CallCount = 0;
      return (NULL);
   }

   if (!IsCgiPlusScript)
   {
      if (!PyArg_ParseTuple (args, "|i", &AllowSingleCGI)) return (NULL);

      if (AllowSingleCGI)
      {
         if (!CallCount++)
         {
            Py_INCREF(Py_True);
            return (Py_True);
         }
      }

      Py_INCREF(Py_False);
      return (Py_False);
   }

   if (CgiPlusPythonEnd)
   {
      if (!IsCgiPlusScript)
      {
         Py_INCREF(Py_False);
         return (Py_False);
      }

      fflush (stdout);
      fputs (CgiPlusEofPtr, stdout);
      fflush (stdout);

      if (!PyArg_ParseTuple (args, "|ii", &AllowSingleCGI, &CheckCodeMtime))
         return (NULL);

      if (CheckCodeMtime)
      {
         if (!ByteCodeUpToDate (CodeCachePtr))
         {
            Py_INCREF(Py_False);
            return (Py_False);
         }
      }

      CgiPlusPythonEnd = 0;
      RemoveOsEnvCgiVar ();
   }

   /* block waiting for the next request (if not the first usage) */
   if (CallCount++) CgiVar ("");

   UsageCount++;
   CodeCachePtr->CodeUsageCount++;
   CgiPlusPythonEnd = 1;

   if (!GlobalDebug && ((cptr = TrnLnm ("PYRTE$DBUG", NULL)) != NULL))
      if (sptr = CgiVar ("SCRIPT_NAME"))
         if (strstr (sptr, cptr))
            if (Debug = 1)
               fprintf (stdout, "Content-Type: text/plain\n\n");

   RteScriptParam ();

   if (!NoStreamMode) fprintf (stdout, "Script-Control: X-stream-mode\n");

   if (!SetupOsEnvCgiVar ())
   {
      Py_INCREF(Py_False);
      return (Py_False);
   }

   Py_INCREF(Py_True);
   return (Py_True);
}

/*****************************************************************************/
/*
It's possible to use this function to explicity end a CGIplus script's request
processing.  If it's not used then the cgiplus_begin() function does all of
this instead.

It accepts zero or one parameters.  The parameter is a boolean (integer 0 or
1).  If True it gets the script code modification time to check for changes. 
If modified it return False.
*/

static PyObject* RtePyMethCgiPlusEnd
(
PyObject *self,
PyObject *args
)
{
   int  CheckCodeMtime = 0;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RtePyMethCgiPlusEnd()\n");

   if (!IsCgiPlusScript)
   {
      Py_INCREF(Py_False);
      return (Py_False);
   }

   fflush (stdout);
   fputs (CgiPlusEofPtr, stdout);
   fflush (stdout);

   CgiPlusPythonEnd = 0;
   RemoveOsEnvCgiVar ();

   if (!PyArg_ParseTuple (args, "|i", &CheckCodeMtime)) return (NULL);

   if (CheckCodeMtime)
   {
      if (!ByteCodeUpToDate (CodeCachePtr))
      {
         Py_INCREF(Py_False);
         return (Py_False);
      }
   }

   Py_INCREF(Py_True);
   return (Py_True);
}

/*****************************************************************************/
/*
Set whether the Python interpreter being used is disposed of at the end of the
current request (0 or False) or resued for the next request (1 or True).
*/

static PyObject* RtePyMethReuseInterpreter
(
PyObject *self,
PyObject *args
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RtePyMethReuseInterpreter()\n");

   if (!PyArg_ParseTuple (args, "i", &ReuseInterpreter)) return (NULL);
   Py_INCREF(Py_None);
   return (Py_None);
}

/*****************************************************************************/
/*
Return a string object containing details of the pyRTE version and build date.
*/

static PyObject* RtePyMethRteId ()

{
   char  RteIdBuf [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RtePyMethRteId()\n");

   sprintf (RteIdBuf, "%s, %s", SOFTWAREID, BUILD_DATETIME);
   return (Py_BuildValue ("s", RteIdBuf));
}

/*****************************************************************************/
/*
Return a string object containing details of the particular cache entry as a
series of comma-separated-values (CSV) in the same order as the data structure.
Note that this function will not succeed without logical name PYRTE_CACHE_ENTRY
being defined.
*/

static PyObject* RtePyMethRteCacheEntry
(
PyObject *self,
PyObject *args
)
{
   static int  idx = 0;

   char  StringBuf [512];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RtePyMethRteCacheEntry()\n");

   if (!TrnLnm ("PYRTE_CACHE_ENTRY", NULL))
   {
      PyErr_SetString (PyExc_RuntimeError,
                       "logical name PYRTE_CACHE_ENTRY not defined");
      return (NULL);
   }

   if (!PyArg_ParseTuple (args, "|i", &idx)) return (NULL);

   for (;;)
   {
      if (idx < 0 || idx >= CodeCacheCount)
      {
         idx = 0;
         Py_INCREF(Py_None);
         return (Py_None);
      }
      if (CodeCache[idx].FileNamePtr) break;
      idx++; 
   }

   sprintf (StringBuf, "%u,%u,%u,%u,%s,%s,%u,%u",
            CodeCache[idx].CodeUsageCount, 
            CodeCache[idx].InterpreterUsageCount, 
            CodeCache[idx].LastUsedSecs, 
            CodeCache[idx].ScriptMtimeSecs, 
            CodeCache[idx].FileNamePtr, 
            CodeCache[idx].FileNameByteCodePtr ? 
               CodeCache[idx].FileNameByteCodePtr : "(null)", 
            (unsigned long)CodeCache[idx].pInterpState, 
            (unsigned long)CodeCache[idx].pByteCode); 

   idx++;

   return (Py_BuildValue ("s", StringBuf));
}

/*****************************************************************************/
/*
Jean-Franois Pironne's original '_wasd' module functions (massaged just a
little :-)
*/

static PyObject* wasd_cgi_init ()
{
   Py_INCREF(Py_None);
   return Py_None;
}

static PyObject* wasd_isCgiPlus
(
PyObject *self,
PyObject *args
)
{
   if (!PyArg_NoArgs(args)) return (NULL);

   if (IsCgiPlusScript)
      return (PyObject *)PyInt_FromLong(1L);
   else
   {
      Py_INCREF(Py_None);
      return Py_None;
   }
}

static PyObject* wasd_cgi_eof
(
PyObject *self,
PyObject *args
)
{
   if (!PyArg_NoArgs(args)) return (NULL);

   if (IsCgiPlusScript)
   {
      fflush(stdout);
      fprintf (stdout, "%s\n", CgiPlusEofPtr);
      fflush(stdout);
   }

   Py_INCREF(Py_None);
   return (Py_None);
}

static PyObject* wasd_cgi_info
(
PyObject *self,
PyObject *args
)
{
   int s;
   char* buffer;
   char* name;
   PyObject* res;

   if (!PyArg_ParseTuple (args, "s", &name)) return (NULL);

   buffer = CgiVar(name);

   if (buffer == NULL)
   {
      Py_INCREF(Py_None);
      res = Py_None;
   }
   else
      res = PyString_FromString(buffer);

   return (res);
}

static PyObject* wasd_cgi_read
(
PyObject *self,
PyObject *args
)
{
   int  retval,
        BufferSize = 0;
   char  *cptr,
         *BodyPtr = NULL;
   int  BodyLength;
   PyObject*  pBody;

   if (!PyArg_ParseTuple(args, "i", &BufferSize)) return (NULL);

   cptr = CgiVar("CONTENT_LENGTH");
   if (cptr && isdigit(*cptr)) BufferSize = atoi(cptr);

   BodyPtr = cptr = calloc (1, BufferSize+32);
   if (!BodyPtr) return (NULL);
   while (BufferSize)
   {
      retval = read (fileno(stdin), BodyPtr, BufferSize);
      if (retval <= 0) break;
      cptr += retval;
      BufferSize -= retval;
   }
   BodyLength = cptr - BodyPtr;

   if (BodyLength == 0)
      pBody = PyString_FromString("");
   else
   if (BodyLength == strlen (BodyPtr))
      pBody = PyString_FromString (BodyPtr);
   else
      pBody = PyString_FromStringAndSize (BodyPtr, BodyLength);

   if (BodyPtr) free (BodyPtr);

   return (pBody);
}

/*****************************************************************************/
/*
*/

void RteScriptParam ()
       
{
   char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RteScriptParam()\n");

   NoStreamMode = 0;

   cptr = CgiVar ("PYRTE");
   if (!cptr) return;
   while (*cptr)
   {
      while (*cptr && *cptr != '/') cptr++;
      if (!*cptr) break;
      if (!strncmp (cptr, "/reuse", 6) || !strncmp (cptr, "/REUSE", 6)) 
         ReuseInterpreter = 1;
      else
      if (!strncmp (cptr, "/noreuse", 8) || !strncmp (cptr, "/NOREUSE", 8)) 
         ReuseInterpreter = 0;
      else
      if (!strncmp (cptr, "/nostream", 9) || !strncmp (cptr, "/NOSTREAM", 9)) 
         NoStreamMode = 1;
      else
      if (!strncmp (cptr, "/dbug", 5) || !strncmp (cptr, "/DBUG", 5)) 
         Debug = 1;
      cptr++;
   }
}

/*****************************************************************************/
/*
Make it look a bit like the standard WASD error message.
*/

void ReportError
(
int SourceCodeLine,
int FormatError,
int VmsStatus,
int PythonErrorPrint,
char *FaoString,
...
)
{
   int  argcnt, status;
   unsigned short  ShortLen;
   char  *cptr, *sptr;
   char  Buffer [2048],
         FaoBuffer [1024],
         Format [256];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (FaoBufferDsc, FaoBuffer);
   $DESCRIPTOR (FaoStringDsc, "");
   unsigned long  *vecptr;
   unsigned long  FaoVector [32];
   va_list  argptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ReportError() %d\n", SourceCodeLine);

   va_count (argcnt);
   vecptr = FaoVector;
   va_start (argptr, FaoString);
   for (argcnt -= 1; argcnt; argcnt--)
      *vecptr++ = (unsigned long)va_arg (argptr, unsigned long);
   va_end (argptr);

   if (FaoString)
   {
      FaoStringDsc.dsc$a_pointer = FaoString;
      FaoStringDsc.dsc$w_length = strlen(FaoString);
      status = sys$faol (&FaoStringDsc, &ShortLen, &BufferDsc,
                         (unsigned long*)&FaoVector);
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      if (status & 1)
         Buffer[ShortLen] = '\0';
      else
         sprintf (Buffer, "SYS$FAO() %%X%08.08X", status);
   }
   else
      Buffer[0] = '\0';

   if (!FormatError)
   {
      if (VmsStatus) fprintf (stdout, "\n<!-- %%X%08.08X -->\n", VmsStatus);
      PyErr_Print();
      return;
   }

   fprintf (stdout,
"Status: 502\n\
Content-Type: text/html\n\
\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"line\" CONTENT=\"%d\">\n\
<TITLE>ERROR 502 Bad Gateway</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<FONT SIZE=\"+1\"><B>ERROR 502</B> &nbsp;-&nbsp; \
External agent did not respond (or not acceptably)</FONT>\n",
           SOFTWAREID,
           SourceCodeLine);

   fprintf (stdout, "<!-- %%X%08.08X -->\n", VmsStatus);

   if (Buffer[0]) fprintf (stdout, "<P>%s", Buffer);

   if (PythonErrorPrint)
   {
      fprintf (stdout, "<P><PRE>");
      PyErr_Print();
      fprintf (stdout, "</PRE>\n");
   }

   cptr = CgiVar ("SERVER_SIGNATURE");

   fprintf (stdout,
"<P><HR WIDTH=85%% ALIGN=left SIZE=2 NOSHADE>\n\
%s%s\
</BODY>\n\
</HTML>\n",
            cptr ? cptr : "", cptr ? "\n" : "");
}

/*****************************************************************************/
/*
Return the value of a CGI variable regardless of whether it is used in a
standard CGI environment or a WASD CGIplus (RTE) environment.  Also
automatically switches WASD V7.2 and later servers into 'struct' mode for
significantly improved performance.

WASD by default supplies CGI variables prefixed by "WWW_" to differentiate them
from any other DCL symbols (or "env"ironment logicals).  Python scripts expect
CGI variables without this.  This function is somewhat tailored to the Python
environment to allow for this, ignoring it with the CGIplus streamed variable,
adding it when access DCL symbols.

Returns a pointer to an internal string if the variable exists, or NULL.
*/

char* CgiVar (char *VarName)

{
#  ifndef CGIVAR_STRUCT_SIZE
#     define CGIVAR_STRUCT_SIZE 8192
#  endif
#  define SOUS sizeof(unsigned short)

   static int  CalloutDone,
               StructLength;
   static char  *NextVarNamePtr;
   static char  StructBuffer [CGIVAR_STRUCT_SIZE];
   static FILE  *CgiPlusIn;
   
   int  status;
   int  Length;
   char  *bptr, *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug && VarName && VarName[0])
      fprintf (stdout, "CgiVar() |%s|\n", !VarName ? "NULL" : VarName);

   if (!VarName || !VarName[0])
   {
      /* initialize */
      StructLength = 0;
      NextVarNamePtr = StructBuffer;
      if (!VarName) return (NULL);
   }

   if (VarName[0])
   {
      /***************************/
      /* return a variable value */
      /***************************/

      if (VarName[0] == '*')
      {
         if (IsCgiPlusMode)
         {
            /* return each CGIplus variable in successive calls */
            if (!(Length = *(unsigned short*)NextVarNamePtr))
            {
               NextVarNamePtr = StructBuffer;
               return (NULL);
            }
            sptr = (NextVarNamePtr += SOUS);
            NextVarNamePtr += Length;
            /* by default CGI variable names are prefixed by "WWW_", ignore */
            return (sptr + 4);
         }

         /* standard CGI */
         return (CgiVarDclSymbol (VarName));
      }

      if (IsCgiPlusMode)
      {
         /* hmmm, CGIplus not initialized */
         if (!StructLength) return (NULL);

         /* return a pointer to this CGIplus variable's value */
         for (bptr=StructBuffer; Length=*(unsigned short*)bptr; bptr+=Length)
         {
            /* by default CGI variable names are prefixed by "WWW_", ignore */
            sptr = (bptr += SOUS) + 4;
            for (cptr = VarName; *cptr && *sptr && *sptr != '='; cptr++, sptr++)
               if (toupper(*cptr) != toupper(*sptr)) break;
            /* if found return a pointer to the value */
            if (!*cptr && *sptr == '=') return (sptr + 1);
         }
         /* not found */
         return (NULL);
      }

      /* standard CGI */
      return (CgiVarDclSymbol (VarName));
   }

   /*****************************/
   /* get the CGIplus variables */
   /*****************************/

   /* cannot "sync" in a non-CGIplus environment */
   if (!VarName[0] && !IsCgiPlusMode)
   {
      StatTimer (FALSE);
      return (NULL);
   }

   /* the CGIPLUSIN stream can be left open */
   if (!CgiPlusIn)
      if (!(CgiPlusIn = fopen (TrnLnm("CGIPLUSIN", NULL), "r")))
         exit (vaxc$errno);

   /* get the starting record (the essentially discardable one) */
   for (;;)
   {
      cptr = fgets (StructBuffer, sizeof(StructBuffer), CgiPlusIn);
      if (!cptr) exit (vaxc$errno);
      /* if the starting sentinal is detected then break */
      if (*(unsigned short*)cptr == '!\0' ||
          *(unsigned short*)cptr == '!\n' ||
          (*(unsigned short*)cptr == '!!' && isdigit(*(cptr+2)))) break;
   }

   StatTimer (FALSE);

   /* detect the CGIplus "force" record-mode environment variable (once) */
   if (*(unsigned short*)cptr == '!!')
   {
      /********************/
      /* CGIplus 'struct' */
      /********************/

      /* get the size of the binary structure */
      StructLength = atoi(cptr+2);
      if (StructLength <= 0 || StructLength > sizeof(StructBuffer))
         exit (SS$_BUGCHECK);

      if (!fread (StructBuffer, 1, StructLength, CgiPlusIn))
         exit (vaxc$errno);
   }
   else
   {
      /*********************/
      /* CGIplus 'records' */
      /*********************/

      /* reconstructs the original 'struct'ure from the records */
      sptr = (bptr = StructBuffer) + sizeof(StructBuffer);
      while (fgets (bptr+SOUS, sptr-(bptr+SOUS), CgiPlusIn))
      {
         /* first empty record (line) terminates variables */
         if (bptr[SOUS] == '\n') break;
         /* note the location of the length word */
         cptr = bptr;
         for (bptr += SOUS; *bptr && *bptr != '\n'; bptr++);
         if (*bptr != '\n') exit (SS$_BUGCHECK);
         *bptr++ = '\0';
         if (bptr >= sptr) exit (SS$_BUGCHECK);
         /* update the length word */
         *(unsigned short*)cptr = bptr - (cptr + SOUS);
      }
      if (bptr >= sptr) exit (SS$_BUGCHECK);
      /* terminate with a zero-length entry */
      *(unsigned short*)bptr = 0;
      StructLength = (bptr + SOUS) - StructBuffer;
   }

   if (!CalloutDone)
   {
      /* provide the CGI callout to set CGIplus into 'struct' mode */
      fflush (stdout);
      fputs (CgiPlusEscPtr, stdout);
      fflush (stdout);
      /* the leading '!' indicates we're not going to read the response */
      fputs ("!CGIPLUS: struct", stdout);
      fflush (stdout);
      fputs (CgiPlusEotPtr, stdout);
      fflush (stdout);
      /* don't need to do this again (the '!!' tells us what mode) */
      CalloutDone = 1;
   }

   return (NULL);

#  undef SOUS
}

/*****************************************************************************/
/*
Standard CGI environment.
Clunky, but what else can we do with DCL symbols?
*/

char* CgiVarDclSymbol (char *VarName)

{
   static char  *SymbolNames [] = {

/* 'standard' CGI names */

"AUTH_ACCESS", "AUTH_AGENT", "AUTH_GROUP", "AUTH_PASSWORD",
"AUTH_REALM", "AUTH_REALM_DESCRIPTION", "AUTH_REMOTE_USER",
"AUTH_TYPE", "AUTH_USER",  "CONTENT_LENGTH", "CONTENT_TYPE",
"DOCUMENT_ROOT", 
/** "GATEWAY_BG", "GATEWAY_EOF", "GATEWAY_EOT", "GATEWAY_ESC", **/
"GATEWAY_INTERFACE", "GATEWAY_MRS",
"HTML_BODYTAG", "HTML_FOOTER", "HTML_FOOTERTAG",
"HTML_HEADER", "HTML_HEADERTAG",
"HTTP_ACCEPT", "HTTP_ACCEPT_CHARSET", "HTTP_ACCEPT_ENCODING", 
"HTTP_ACCEPT_LANGUAGE", "HTTP_AUTHORIZATION", "HTTP_CONNECTION",
"HTTP_CACHE_CONTROL", "HTTP_CONTENT_LENGTH", "HTTP_CONTENT_ENCODING",
"HTTP_CONTENT_TYPE", "HTTP_COOKIE", "HTTP_FORWARDED", "HTTP_HOST",
"HTTP_IF_MATCH", "HTTP_IF_NOT_MODIFIED", "HTTP_IF_NONE_MATCH",
"HTTP_IF_MODIFIED_SINCE", "HTTP_IF_UNMODIFIED_SINCE", "HTTP_IF_RANGE",
"HTTP_KEEP_ALIVE", "HTTP_RANGE", "HTTP_TRAILER", "HTTP_TRANSFER_ENCODING",
"HTTP_PRAGMA", "HTTP_REFERER", "HTTP_USER_AGENT", "HTTP_X_FORWARDED_FOR",
"PATH_INFO", "PATH_ODS", "PATH_TRANSLATED",
"QUERY_STRING", "REMOTE_ADDR", "REMOTE_HOST", "REMOTE_PORT", "REMOTE_USER", 
"REQUEST_CHARSET", "REQUEST_CONTENT_TYPE", "REQUEST_METHOD",
"REQUEST_SCHEME", "REQUEST_TIME_GMT", "REQUEST_TIME_LOCAL", "REQUEST_URI",
"SCRIPT_DEFAULT", "SCRIPT_FILENAME", "SCRIPT_NAME", "SCRIPT_RTE",
"SERVER_ADDR", "SERVER_ADMIN", "SERVER_CHARSET", "SERVER_GMT", "SERVER_NAME",
"SERVER_PROTOCOL", "SERVER_PORT", "SERVER_SOFTWARE", "SERVER_SIGNATURE",
"UNQIUE_ID",

/* Apache mod_ssl names */

"HTTPS", "SSL_PROTOCOL", "SSL_SESSION_ID", "SSL_CIPHER", "SSL_CIPHER_EXPORT",
"SSL_CIPHER_USEKEYSIZE", "SSL_CIPHER_ALGKEYSIZE", "SSL_CLIENT_M_VERSION",
"SSL_CLIENT_M_SERIAL", "SSL_CLIENT_S_DN", "SSL_CLIENT_S_DN_x509",
"SSL_CLIENT_I_DN", "SSL_CLIENT_I_DN_x509", "SSL_CLIENT_V_START",
"SSL_CLIENT_V_END", "SSL_CLIENT_A_SIG", "SSL_CLIENT_A_KEY", "SSL_CLIENT_CERT",
"SSL_SERVER_M_VERSION", "SSL_SERVER_M_SERIAL", "SSL_SERVER_S_DN",
"SSL_SERVER_S_DN_x509", "SSL_SERVER_I_DN", "SSL_SERVER_I_DN_x509",
"SSL_SERVER_V_START", "SSL_SERVER_V_END", "SSL_SERVER_A_SIG",
"SSL_SERVER_A_KEY", "SSL_SERVER_CERT", "SSL_VERSION_INTERFACE",
"SSL_VERSION_LIBRARY",

/* Purveyor SSL names */

"SECURITY_STATUS", "SSL_CIPHER", "SSL_CIPHER_KEYSIZE", "SSL_CLIENT_CA",
"SSL_CLIENT_DN", "SSL_SERVER_CA", "SSL_SERVER_DN", "SSL_VERSION",

/* X509 certificate names */

"AUTH_X509_CIPHER", "AUTH_X509_FINGERPRINT", "AUTH_X509_ISSUER",
"AUTH_X509_KEYSIZE", "AUTH_X509_SUBJECT",

/* a NULL to terminate the names */

NULL };

   static int  SymbolIndex = 0;
   static char  WwwName [256] = "WWW_";
   static $DESCRIPTOR (NameDsc, WwwName);
   static $DESCRIPTOR (ValueDsc, "");

   int  status;
   unsigned short  ShortLength;
   char  *cptr;
   char  Value [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CgiVarDclSymbol()\n");

   if (VarName[0] == '*')
      cptr = SymbolNames[SymbolIndex++];
   else
   {
      cptr = VarName;
      SymbolIndex = 0;
   }

   while (cptr)
   {
      /* by default WASD CGI variable names are prefixed by "WWW_", add */
      strncpy (WwwName+4, cptr, sizeof(WwwName)-5);
      NameDsc.dsc$w_length = strlen(WwwName);
      ValueDsc.dsc$a_pointer = Value;
      ValueDsc.dsc$w_length = 1023;
   
      status = lib$get_symbol (&NameDsc, &ValueDsc, &ShortLength, NULL);
      if (status & 1)
      {
         cptr = malloc (ShortLength+1);
         memcpy (cptr, Value, ShortLength);
         cptr[ShortLength] = '\0';
         return (cptr);
      }

      if (VarName[0] != '*') return (NULL);
      /* assume the failure was 'no such symbol' and look further */
      cptr = SymbolNames[SymbolIndex++];
   }

   SymbolIndex = 0;
   return (NULL);
}

/*****************************************************************************/
/*
If false then initialize the LIB$ stats timer.
If true return a pointer to a static buffer containings a stats string.
*/

char* StatTimer (int ShowStat)

{
   static $DESCRIPTOR (FaoDsc,
"REAL:!%T CPU:!UL.!2ZL DIO:!UL BIO:!UL FAULTS:!UL\0");

   static char  StatString [96];
   static $DESCRIPTOR (StatStringDsc, StatString);

   static unsigned long  LibStatTimerReal = 1,
                         LibStatTimerCpu = 2,
                         LibStatTimerBio = 3,
                         LibStatTimerDio = 4,
                         LibStatTimerFaults = 5;

   int  status;
   unsigned long  CpuBinTime,
                  CountBio,
                  CountDio,
                  CountFaults;
   unsigned long  RealBinTime [2];

   /*********/
   /* begin */
   /*********/

   if (!ShowStat)
   {
      lib$init_timer (0);
      return (NULL);
   }

   /* post-processing reset */
   lib$stat_timer (&LibStatTimerReal, &RealBinTime, 0);
   lib$stat_timer (&LibStatTimerCpu, &CpuBinTime, 0);
   lib$stat_timer (&LibStatTimerBio, &CountBio, 0);
   lib$stat_timer (&LibStatTimerDio, &CountDio, 0);
   lib$stat_timer (&LibStatTimerFaults, &CountFaults, 0);
   sys$fao (&FaoDsc, 0, &StatStringDsc,
            &RealBinTime, CpuBinTime/100, CpuBinTime%100,
            CountDio, CountBio, CountFaults);

   return (StatString);
}

/*****************************************************************************/
/*
Translate a logical name using LNM$FILE_DEV.  Returns a pointer to the value
string, or NULL if the name does not exist.  If 'LogValue' is supplied the
logical name is translated into that (assumed to be large enough), otherwise
it's translated into an internal static buffer.
*/

char* TrnLnm
(
char *LogName,
char *LogValue
)
{
   static unsigned short  ShortLength;
   static char  StaticLogValue [256];
   static $DESCRIPTOR (LogNameDsc, "");
   static $DESCRIPTOR (LnmFileDevDsc, "LNM$FILE_DEV");
   static struct {
      short int  buf_len;
      short int  item;
      void  *buf_addr;
      unsigned short  *ret_len;
   } LnmItems [] =
   {
      { 255, LNM$_STRING, 0, &ShortLength },
      { 0,0,0,0 }
   };

   int  status;
   char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TrnLnm() |%s|\n", LogName);

   LogNameDsc.dsc$a_pointer = LogName;
   LogNameDsc.dsc$w_length = strlen(LogName);
   if (LogValue)
      cptr = LnmItems[0].buf_addr = LogValue;
   else
      cptr = LnmItems[0].buf_addr = StaticLogValue;

   status = sys$trnlnm (0, &LnmFileDevDsc, &LogNameDsc, 0, &LnmItems);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (!(status & 1))
   {
      if (Debug) fprintf (stdout, "|(null)|\n");
      return (NULL);
   }

   cptr[ShortLength] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", cptr);
   return (cptr);
}

/*****************************************************************************/
/*
*/

int SetCrtlFeature
(
char *name,
int value
)
{
    int  idx;

   /*********/
   /* begin */
   /*********/

    if (Debug) fprintf (stdout, "SetCrtlFeature() |%s| %d\n", name, value);

    idx = decc$feature_get_index(name);
    if (idx == -1)
    {
       if (Debug) perror(name);
       return (-1);
    }
    if (decc$feature_set_value(idx, 1, value) == -1)
    {
       if (Debug) perror(name);
       return (-1);
    }
    if (decc$feature_get_value(idx, 1) != value)
    {
       if (Debug) perror(name);
       return (-1);
    }
    return (0);
}

/*****************************************************************************/

